/* texindex -- sort TeX index dribble output into an actual index.
   $Id: texindex.c 5191 2013-02-23 00:11:18Z karl $

   Copyright 1987, 1991, 1992, 1996, 1997, 1998, 1999, 2000, 2001,
   2002, 2003, 2004, 2005, 2006, 2007, 2008, 2009, 2010, 2011, 2012,
   2013 Free Software Foundation, Inc.

   This program is free software: you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.  */

#include "system.h"
#include <getopt.h>

#ifdef WIN32
#include <string.h>
#include <fcntl.h>
#undef va_start
#undef va_end
#undef va_arg
#include <stdarg.h>
#define const
#endif

char *program_name = "texindex";

#if defined (emacs)
#  include "../src/config.h"
/* Some s/os.h files redefine these. */
#  undef read
#  undef close
#  undef write
#  undef open
#endif

#if !defined (HAVE_MEMSET)
#undef memset
#define memset(ptr, ignore, count) bzero (ptr, count)
#endif

#ifndef WIN32
char *mktemp (char *);
#endif

#if !defined (SEEK_SET)
#  define SEEK_SET 0
#  define SEEK_CUR 1
#  define SEEK_END 2
#endif /* !SEEK_SET */

/* When sorting in core, this structure describes one line
   and the position and length of its first keyfield.  */
struct lineinfo
{
  char *text;           /* The actual text of the line. */
  union {
    char *text;         /* The start of the key (for textual comparison). */
    long number;        /* The numeric value (for numeric comparison). */
  } key;
  long keylen;          /* Length of KEY field. */
};

/* This structure describes a field to use as a sort key. */
struct keyfield
{
  int startwords;       /* Number of words to skip. */
  int startchars;       /* Number of additional chars to skip. */
  int endwords;         /* Number of words to ignore at end. */
  int endchars;         /* Ditto for characters of last word. */
  char ignore_blanks;   /* Non-zero means ignore spaces and tabs. */
  char fold_case;       /* Non-zero means case doesn't matter. */
  char reverse;         /* Non-zero means compare in reverse order. */
  char numeric;         /* Non-zeros means field is ASCII numeric. */
  char positional;      /* Sort according to file position. */
  char braced;          /* Count balanced-braced groupings as fields. */
};

/* Vector of keyfields to use. */
struct keyfield keyfields[3];

/* Number of keyfields stored in that vector.  */
int num_keyfields = 3;

/* Vector of input file names, terminated with a null pointer. */
char **infiles;

/* Vector of corresponding output file names, or NULL, meaning default it
   (add an `s' to the end). */
char **outfiles;

/* Length of `infiles'. */
int num_infiles;

/* Pointer to the array of pointers to lines being sorted. */
char **linearray;

/* The allocated length of `linearray'. */
long nlines;

/* During in-core sort, this points to the base of the data block
   which contains all the lines of data.  */
char *text_base;

/* Initially 0; changed to 1 if we want initials in this index.  */
int need_initials;

/* Remembers the first initial letter seen in this index, so we can
   determine whether we need initials in the sorted form.  */
char first_initial;

/* Forward declarations of functions in this file. */
void decode_command (int argc, char **argv);
void sort_in_core (char *infile, int total, char *outfile);
char **parsefile (char *filename, char **nextline, char *data, long int size);
char *find_field (struct keyfield *keyfield, char *str, long int *lengthptr);
char *find_pos (char *str, int words, int chars, int ignore_blanks);
long find_value (char *start, long int length);
char *find_braced_pos (char *str, int words, int chars, int ignore_blanks);
char *find_braced_end (char *str);
void writelines (char **linearray, int nlines, FILE *ostream);
int compare_field (struct keyfield *keyfield, char *start1,
                   long int length1, long int pos1, char *start2,
                   long int length2, long int pos2);
int compare_full (const void *, const void *);
void pfatal_with_name (const char *name);
void fatal (const char *format, ...);
void error (const char *format, ...);
void *xmalloc (), *xrealloc ();
static char *concat3 (const char *, const char *, const char *);

int
main (int argc, char *argv[])
{
  int i;

#ifdef WIN32
  strcpy(argv[0], "texindex");
  setmode(fileno(stdout), _O_BINARY);
#endif
#ifdef HAVE_SETLOCALE
  /* Set locale via LC_ALL.  */
  setlocale (LC_ALL, "");
#endif

  /* Set the text message domain.  */
  bindtextdomain (PACKAGE, LOCALEDIR);
  textdomain (PACKAGE);

  /* In case we write to a redirected stdout that fails.  */
  /* not ready atexit (close_stdout); */

  /* Describe the kind of sorting to do. */
  /* The first keyfield uses the first braced field and folds case. */
  keyfields[0].braced = 1;
  keyfields[0].fold_case = 1;
  keyfields[0].endwords = -1;
  keyfields[0].endchars = -1;

  /* The second keyfield uses the second braced field, numerically. */
  keyfields[1].braced = 1;
  keyfields[1].numeric = 1;
  keyfields[1].startwords = 1;
  keyfields[1].endwords = -1;
  keyfields[1].endchars = -1;

  /* The third keyfield (which is ignored while discarding duplicates)
     compares the whole line. */
  keyfields[2].endwords = -1;
  keyfields[2].endchars = -1;

  decode_command (argc, argv);

  /* Process input files completely, one by one.  */

  for (i = 0; i < num_infiles; i++)
    {
      int desc;
      off_t ptr;
      char *outfile;
      struct stat instat;

      desc = open (infiles[i], O_RDONLY, 0);
      if (desc < 0)
        pfatal_with_name (infiles[i]);

      if (stat (infiles[i], &instat))
        pfatal_with_name (infiles[i]);
      if (S_ISDIR (instat.st_mode))
        {
#ifdef EISDIR
          errno = EISDIR;
#endif
          pfatal_with_name (infiles[i]);
        }

      lseek (desc, (off_t) 0, SEEK_END);
      ptr = (off_t) lseek (desc, (off_t) 0, SEEK_CUR);

      close (desc);

      outfile = outfiles[i];
      if (!outfile)
        outfile = concat3 (infiles[i], "s", "");

      need_initials = 0;
      first_initial = '\0';

      if (ptr != (int)ptr)
	{
	  fprintf (stderr, "%s: %s: file too large\n", program_name,
		   infiles[i]);
	  exit (EXIT_FAILURE);
	}
      sort_in_core (infiles[i], (int)ptr, outfile);
    }

  exit (EXIT_SUCCESS);
}

typedef struct
{
  char *long_name;
  char *short_name;
  int *variable_ref;
  int variable_value;
  char *arg_name;
  char *doc_string;
} TEXINDEX_OPTION;

TEXINDEX_OPTION texindex_options[] = {
  { "--help", "-h", (int *)NULL, 0, (char *)NULL,
      N_("display this help and exit") },
  { "--output", "-o", (int *)NULL, 0, "FILE",
      N_("send output to FILE") },
  { "--version", (char *)NULL, (int *)NULL, 0, (char *)NULL,
      N_("display version information and exit") },
  { (char *)NULL, (char *)NULL, (int *)NULL, 0, (char *)NULL }
};

void
usage (int result_value)
{
  register int i;
  FILE *f = result_value ? stderr : stdout;

  fprintf (f, _("Usage: %s [OPTION]... FILE...\n"), program_name);
  fprintf (f, _("Generate a sorted index for each TeX output FILE.\n"));
  /* Avoid trigraph nonsense.  */
  fprintf (f,
_("Usually FILE... is specified as `foo.%c%c\' for a document `foo.texi'.\n"),
           '?', '?'); /* avoid trigraph in cat-id-tbl.c */
  fprintf (f, _("\nOptions:\n"));

  for (i = 0; texindex_options[i].long_name; i++)
    {
      putc (' ', f);

      if (texindex_options[i].short_name)
        fprintf (f, "%s, ", texindex_options[i].short_name);

      fprintf (f, "%s %s",
               texindex_options[i].long_name,
               texindex_options[i].arg_name
               ? texindex_options[i].arg_name : "");

      fprintf (f, "\t%s\n", _(texindex_options[i].doc_string));
    }
  fputs (_("\n\
Email bug reports to bug-texinfo@gnu.org,\n\
general questions and discussion to help-texinfo@gnu.org.\n\
Texinfo home page: http://www.gnu.org/software/texinfo/"), f);
  fputs ("\n", f);

  exit (result_value);
}

/* Decode the command line arguments to set the parameter variables
   and set up the vector of keyfields and the vector of input files. */

void
decode_command (int argc, char **argv)
{
  int arg_index = 1;
  char **ip;
  char **op;

  /* Allocate ARGC input files, which must be enough.  */

  infiles = (char **) xmalloc (argc * sizeof (char *));
  outfiles = (char **) xmalloc (argc * sizeof (char *));
  ip = infiles;
  op = outfiles;

  while (arg_index < argc)
    {
      char *arg = argv[arg_index++];

      if (*arg == '-')
        {
          if (strcmp (arg, "--version") == 0)
            {
              printf ("texindex (GNU %s) %s\n", PACKAGE, VERSION);
              puts ("");
              printf (_("Copyright (C) %s Free Software Foundation, Inc.\n\
License GPLv3+: GNU GPL version 3 or later <http://gnu.org/licenses/gpl.html>\n\
This is free software: you are free to change and redistribute it.\n\
There is NO WARRANTY, to the extent permitted by law.\n"),
	      "2013");
              exit (EXIT_SUCCESS);
            }
          else if ((strcmp (arg, "--keep") == 0) ||
                   (strcmp (arg, "-k") == 0))
            {
	      /* Ignore, for backward compatibility */
            }
          else if ((strcmp (arg, "--help") == 0) ||
                   (strcmp (arg, "-h") == 0))
            {
              usage (EXIT_SUCCESS);
            }
          else if ((strcmp (arg, "--output") == 0) ||
                   (strcmp (arg, "-o") == 0))
            {
              if (argv[arg_index] != (char *)NULL)
                {
                  arg_index++;
                  if (op > outfiles)
                    *(op - 1) = argv[arg_index];
                }
              else
                usage (EXIT_FAILURE);
            }
          else
            usage (EXIT_FAILURE);
        }
      else
        {
          *ip++ = arg;
          *op++ = (char *)NULL;
        }
    }

  /* Record number of keyfields and terminate list of filenames. */
  num_infiles = ip - infiles;
  *ip = (char *)NULL;
  if (num_infiles == 0)
    usage (EXIT_FAILURE);
}

/* Compare LINE1 and LINE2 according to the specified set of keyfields. */

int
compare_full (const void *p1, const void *p2)
{
  char **line1 = (char **) p1;
  char **line2 = (char **) p2;
  int i;

  /* Compare using the first keyfield;
     if that does not distinguish the lines, try the second keyfield;
     and so on. */

  for (i = 0; i < num_keyfields; i++)
    {
      long length1, length2;
      char *start1 = find_field (&keyfields[i], *line1, &length1);
      char *start2 = find_field (&keyfields[i], *line2, &length2);
      int tem = compare_field (&keyfields[i], start1, length1,
                               *line1 - text_base,
                               start2, length2, *line2 - text_base);
      if (tem)
        {
          if (keyfields[i].reverse)
            return -tem;
          return tem;
        }
    }

  return 0;                     /* Lines match exactly. */
}

/* Compare LINE1 and LINE2, described by structures
   in which the first keyfield is identified in advance.
   For positional sorting, assumes that the order of the lines in core
   reflects their nominal order.  */
int
compare_prepared (const void *p1, const void *p2)
{
  struct lineinfo *line1 = (struct lineinfo *) p1;
  struct lineinfo *line2 = (struct lineinfo *) p2;
  int i;
  int tem;
  char *text1, *text2;

  /* Compare using the first keyfield, which has been found for us already. */
  if (keyfields->positional)
    {
      if (line1->text - text_base > line2->text - text_base)
        tem = 1;
      else
        tem = -1;
    }
  else if (keyfields->numeric)
    tem = line1->key.number - line2->key.number;
  else
    tem = compare_field (keyfields, line1->key.text, line1->keylen, 0,
                         line2->key.text, line2->keylen, 0);
  if (tem)
    {
      if (keyfields->reverse)
        return -tem;
      return tem;
    }

  text1 = line1->text;
  text2 = line2->text;

  /* Compare using the second keyfield;
     if that does not distinguish the lines, try the third keyfield;
     and so on. */

  for (i = 1; i < num_keyfields; i++)
    {
      long length1, length2;
      char *start1 = find_field (&keyfields[i], text1, &length1);
      char *start2 = find_field (&keyfields[i], text2, &length2);
      int tem = compare_field (&keyfields[i], start1, length1,
                               text1 - text_base,
                               start2, length2, text2 - text_base);
      if (tem)
        {
          if (keyfields[i].reverse)
            return -tem;
          return tem;
        }
    }

  return 0;                     /* Lines match exactly. */
}

/* Like compare_full but more general.
   You can pass any strings, and you can say how many keyfields to use.
   POS1 and POS2 should indicate the nominal positional ordering of
   the two lines in the input.  */

int
compare_general (char *str1, char *str2, long int pos1, long int pos2, int use_keyfields)
{
  int i;

  /* Compare using the first keyfield;
     if that does not distinguish the lines, try the second keyfield;
     and so on. */

  for (i = 0; i < use_keyfields; i++)
    {
      long length1, length2;
      char *start1 = find_field (&keyfields[i], str1, &length1);
      char *start2 = find_field (&keyfields[i], str2, &length2);
      int tem = compare_field (&keyfields[i], start1, length1, pos1,
                               start2, length2, pos2);
      if (tem)
        {
          if (keyfields[i].reverse)
            return -tem;
          return tem;
        }
    }

  return 0;                     /* Lines match exactly. */
}

/* Find the start and length of a field in STR according to KEYFIELD.
   A pointer to the starting character is returned, and the length
   is stored into the int that LENGTHPTR points to.  */

char *
find_field (struct keyfield *keyfield, char *str, long int *lengthptr)
{
  char *start;
  char *end;
  char *(*fun) ();

  if (keyfield->braced)
    fun = find_braced_pos;
  else
    fun = find_pos;

  start = (*fun) (str, keyfield->startwords, keyfield->startchars,
                  keyfield->ignore_blanks);
  if (keyfield->endwords < 0)
    {
      if (keyfield->braced)
        end = find_braced_end (start);
      else
        {
          end = start;
          while (*end && *end != '\n')
            end++;
        }
    }
  else
    {
      end = (*fun) (str, keyfield->endwords, keyfield->endchars, 0);
      if (end - str < start - str)
        end = start;
    }
  *lengthptr = end - start;
  return start;
}

/* Return a pointer to a specified place within STR,
   skipping (from the beginning) WORDS words and then CHARS chars.
   If IGNORE_BLANKS is nonzero, we skip all blanks
   after finding the specified word.  */

char *
find_pos (char *str, int words, int chars, int ignore_blanks)
{
  int i;
  char *p = str;

  for (i = 0; i < words; i++)
    {
      char c;
      /* Find next bunch of nonblanks and skip them. */
      while ((c = *p) == ' ' || c == '\t')
        p++;
      while ((c = *p) && c != '\n' && !(c == ' ' || c == '\t'))
        p++;
      if (!*p || *p == '\n')
        return p;
    }

  while (*p == ' ' || *p == '\t')
    p++;

  for (i = 0; i < chars; i++)
    {
      if (!*p || *p == '\n')
        break;
      p++;
    }
  return p;
}

/* Like find_pos but assumes that each field is surrounded by braces
   and that braces within fields are balanced. */

char *
find_braced_pos (char *str, int words, int chars, int ignore_blanks)
{
  int i;
  int bracelevel;
  char *p = str;
  char c;

  for (i = 0; i < words; i++)
    {
      bracelevel = 1;
      while ((c = *p++) != '{' && c != '\n' && c)
        /* Do nothing. */ ;
      if (c != '{')
        return p - 1;
      while (bracelevel)
        {
          c = *p++;
#ifdef	SJIS
	  if ( iskanji(c)) {
		  p++;
		  continue;
	  }
#endif
          if (c == '{')
            bracelevel++;
          if (c == '}')
            bracelevel--;
          if (c == 0 || c == '\n')
            return p - 1;
        }
    }

  while ((c = *p++) != '{' && c != '\n' && c)
    /* Do nothing. */ ;

  if (c != '{')
    return p - 1;

  if (ignore_blanks)
    while ((c = *p) == ' ' || c == '\t')
      p++;

  for (i = 0; i < chars; i++)
    {
      if (!*p || *p == '\n')
        break;
      p++;
    }
  return p;
}

/* Find the end of the balanced-brace field which starts at STR.
   The position returned is just before the closing brace. */

char *
find_braced_end (char *str)
{
  int bracelevel;
  char *p = str;
  char c;

  bracelevel = 1;
  while (bracelevel)
    {
      c = *p++;
#ifdef	SJIS
	  if ( iskanji(c)) {
		  p++;
		  continue;
	  }
#endif
      if (c == '{')
        bracelevel++;
      if (c == '}')
        bracelevel--;
      if (c == 0 || c == '\n')
        return p - 1;
    }
  return p - 1;
}

long
find_value (char *start, long int length)
{
  while (length != 0L)
    {
      if (isdigit (*start))
        return atol (start);
      length--;
      start++;
    }
  return 0l;
}

/* Vector used to translate characters for comparison.
   This is how we make all alphanumerics follow all else,
   and ignore case in the first sorting.  */
/*
int char_order[256];
*/
int char_order[0x10000]; /* japanese */

#ifdef	EUC
#define	KANJI
#endif

#ifdef	SJIS
#define	KANJI
#endif

#ifdef		KANJI

#ifndef		EUC
#	ifndef		SJIS
#		define	EUC
#	endif
#else
#	ifdef	SJIS
#		error
#	endif
#endif

#ifdef          EUC
#define		JIS_NUM_0		0xa3b0
#define		JIS_NUM_9		(JIS_NUM_0 + '9'- '0')
#define		JIS_ALPH_A		0xa3c1
#define		JIS_ALPH_Z		(JIS_ALPH_A + 'Z'-'A')
#define		JIS_ALPH_a		0xa3e1
#define		JIS_HIRA_SMALL_A	0xa4a1
#define		JIS_HIRA_NN             0xa4f3
#define		JIS_KATA_SMALL_A        0xa5a1
#define		JIS_KATA_SMALL_KE       0xa5f6
#define		JIS_HIRA_U              0xa4a6
#define		JIS_KATA_U              0xa5a6
#define		JIS_KATA_VU             0xa5f4

#define		JIS_KATA_UPPER		0xa5
#define		JIS_HIRA_UPPER		0xa4

#define		JIS_HIRA_A		0xa4a2
#define		JIS_KATA_A		0xa5a2

#define		JIS_DAKUTEN		0xa1ab
#define		JIS_HANDAKUTEN          0xa1ac
#define		JIS_CHOUON		0xa2ac

int kana[] = {
/*	          .        [      ]      ,        .       wo      a */	
	0xa1a1, 0xa1a3, 0xa1d6, 0xa1d7, 0xa1a2, 0xa1a6, 0xa5f2, 0xa5a1,
/*    i       u       e       o       ya      yu      yo      tsu   */	
	0xa5a3, 0xa5a5, 0xa5a7, 0xa5a9, 0xa5e3, 0xa5e5, 0xa5e7, 0xa5c3,
/*     -       a       i       u       e       o      ka      ki    */
	0xa1bc, 0xa5a2, 0xa5a4, 0xa5a6, 0xa5a8, 0xa5aa, 0xa5ab, 0xa5ad,
/*    ku      ke      ko      sa      si      su     se       so    */
	0xa5af,	0xa5b1, 0xa5b3, 0xa5b5, 0xa5b7, 0xa5b9, 0xa5bb, 0xa5bd,
/*    ta      ti      tu      te      to      na     ni       nu    */	
	0xa5bf, 0xa5c1, 0xa5c4, 0xa5c6, 0xa5c8, 0xa5ca, 0xa5cb, 0xa5cc,
/*    ne      no      ha      hi      hu      he      ho      ma    */
	0xa5cd, 0xa5ce, 0xa5cf, 0xa5d2, 0xa5d5, 0xa5d8, 0xa5db, 0xa5de,
/*    mi      mu      me      mo      ya      yu      yo      ra    */
	0xa5df, 0xa5e0, 0xa5e1, 0xa5e2, 0xa5e4, 0xa5e6, 0xa5e8, 0xa5e9,
/*	  ri      ru      re      ro      wa      nn      ""      maru  */
	0xa5ea, 0xa5eb, 0xa5ec, 0xa5ed, 0xa5ef, 0xa5f3,	JIS_DAKUTEN,JIS_HANDAKUTEN
};

int daku[] = {0xa5ac, 0xa5ae, 0xa5b0, 0xa5b2, 0xa5b4,
              0xa5b6, 0xa5b8, 0xa5ba, 0xa5bc, 0xa5be,
              0xa5c0, 0xa5c2, 0xa5c5, 0xa5c7, 0xa5c9,
              0xa5d0, 0xa5d3, 0xa5d6, 0xa5d9, 0xa5dc};
int handaku[] = {0xa5d1, 0xa5d4, 0xa5d7, 0xa5da, 0xa5dd};
int dakuall[] = {0xa4ac, 0xa4ae, 0xa4b0, 0xa4b2, 0xa4b4,
                 0xa4b6, 0xa4b8, 0xa4ba, 0xa4bc, 0xa4be,
                 0xa4c0, 0xa4c2, 0xa4c5, 0xa4c7, 0xa4c9,
                 0xa4d0, 0xa4d3, 0xa4d6, 0xa4d9, 0xa4dc,
                 0xa5ac, 0xa5ae, 0xa5b0, 0xa5b2, 0xa5b4,
                 0xa5b6, 0xa5b8, 0xa5ba, 0xa5bc, 0xa5be,
                 0xa5c0, 0xa5c2, 0xa5c5, 0xa5c7, 0xa5c9,
                 0xa5d0, 0xa5d3, 0xa5d6, 0xa5d9, 0xa5dc};
int handakuall[] = {0xa4d1, 0xa4d4, 0xa4d7, 0xa4da, 0xa4dd,
                    0xa5d1, 0xa5d4, 0xa5d7, 0xa5da, 0xa5dd};
int
iskanji(int c) /* japanses extention */
{
	if ( !(c & 0x80 )) return 0;
	if ( c == 0x8e ) return 0;
	return 1;
}

int 
is201kana(int c)  /* japanses extention */
{
	if ( c == 0x8e ) return 1;
	else return 0;
}
#else /* SJIS */
#define		JIS_NUM_0		0x824f
#define		JIS_NUM_9		(JIS_NUM_0 + '9'- '0')
#define		JIS_ALPH_A		0x8260
#define		JIS_ALPH_Z		(JIS_ALPH_A + 'Z'-'A')
#define		JIS_ALPH_a		0x8281
#define		JIS_HIRA_SMALL_A	0x829f
#define		JIS_HIRA_NN		0x82f1
#define		JIS_KATA_SMALL_A	0x8340
#define		JIS_KATA_SMALL_KE	0x8396
#define		JIS_HIRA_U		0x82a4
#define		JIS_KATA_U		0x8345 /* org 0x83a5  A.K. 1999 */
#define		JIS_KATA_VU		0x8394

#define		JIS_KATA_UPPER		0x83
#define		JIS_HIRA_UPPER		0x82

#define		JIS_HIRA_A		0x82a0
#define		JIS_KATA_A		0x8341

#define		JIS_DAKUTEN		0x814a
#define		JIS_HANDAKUTEN          0x814b
#define		JIS_CHOUON		0x815b

int kana[] = {
/*	          .        [      ]      ,        .       wo     a  */
	0x813f, 0x8142, 0x8175, 0x8176, 0x8141, 0x8145, 0x8392, 0x8340,
/*    i       u       e       o       ya      yu      yo      tsu   */
	0x8342, 0x8344, 0x8346, 0x8348, 0x8383, 0x8385, 0x8387, 0x8362,
/*     -       a       i       u       e       o      ka      ki    */
	0x815b, 0x8341, 0x8343, 0x8345, 0x8347, 0x8349, 0x834a, 0x834c,
/*    ku      ke      ko      sa      si      su     se       so    */
	0x834e,	0x8350, 0x8352, 0x8354, 0x8356, 0x8358, 0x835a, 0x835c,
/*    ta      ti      tu      te      to      na     ni       nu    */	
	0x835e, 0x8360, 0x8363, 0x8365, 0x8367, 0x8369, 0x836a, 0x836b,
/*    ne      no      ha      hi      hu      he      ho      ma    */
	0x836c, 0x836d, 0x836e, 0x8371, 0x8374, 0x8377, 0x837a, 0x837d,
/*    mi      mu      me      mo      ya      yu      yo      ra    */
	0x837e, 0x8380, 0x8381, 0x8382, 0x8384, 0x8386, 0x8388, 0x8389,
/*	  ri      ru      re      ro      wa      nn      ""  maru  */
	0x838a, 0x838b, 0x838c, 0x838d, 0x838f, 0x8393,	JIS_DAKUTEN,JIS_HANDAKUTEN
};

int daku[] = {0x834b, 0x834d, 0x834f, 0x8351, 0x8353,
              0x8355, 0x8357, 0x8359, 0x835b, 0x835d,
              0x835f, 0x8361, 0x8364, 0x8366, 0x8367,
              0x836f, 0x8372, 0x8375, 0x8378, 0x837b};

int handaku[] = {0x8370, 0x8373, 0x8376, 0x8379, 0x837c};

int dakuall[] = {0x82aa, 0xa2ac, 0x82a3, 0x82b0, 0x82b2,
                 0x82b4, 0x82b6, 0x82b8, 0x82ba, 0x82bc,
                 0x82be, 0x82c0, 0x82c3, 0x82c5, 0x82c7,
                 0x82ce, 0x82d1, 0x82d4, 0x82d7, 0x82da,
                 0x834b, 0x834d, 0x834f, 0x8351, 0x8353,
                 0x8355, 0x8357, 0x8359, 0x835b, 0x835d,
                 0x835f, 0x8361, 0x8364, 0x8366, 0x8367,
                 0x836f, 0x8372, 0x8375, 0x8378, 0x837b};

int handakuall[] = {0x82cf, 0x82d2, 0x82d5, 0x82d8, 0x82db,
                    0x8370, 0x8373, 0x8376, 0x8379, 0x837c};

int
iskanji(int c)   /* japanses extention */
{
	c &= 0xff;
	if (( c >= 0x81 && c <= 0x9f ) ||
		( c >= 0xe0 && c <= 0xea )) return 1;
	return 0;
}

int
is201kana(int c)   /* japanses extention */
{
	if ( c >= 0xa0 && c <= 0xdf ) return 1;
	else return 0;
}
#endif

void
kana_char_order(void) /* japanese */
{
	unsigned int c, i, j, cc;

	for ( c = 0x8000 ; c < 0x10000 ; c++ )
		char_order[c] = c;

	/* NUMERIC */
	for ( c = JIS_NUM_0 , i = '0' + 512; c <= JIS_NUM_9 ; i++,c++)
		char_order[c] = i;

	/* ALPHABET */
	for ( c = JIS_ALPH_A , i = 'a' + 512; c <= JIS_ALPH_Z ; i++,c++) {
		char_order[c] = i;
		char_order[c+JIS_ALPH_a - JIS_ALPH_A] = i;
	}

	/* KANA */
	for ( c = JIS_HIRA_SMALL_A ; c <= JIS_HIRA_NN ; c++ ) {
		char_order[c] = c+512;
		cc = c+JIS_KATA_SMALL_A-JIS_HIRA_SMALL_A;
#ifdef	SJIS
		if ( cc > 0x837e ) cc++;
#endif
		char_order[cc] = c+512;
		if (isdaku(c) ) {
			char_order[c] = c+512-1;
			char_order[cc] = c+512-1;
		} else if (ishandaku(c) ) {
			char_order[c] = c+512-2;
			char_order[cc] = c+512-2;
		}
	}
	for ( c = JIS_KATA_VU ; c <= JIS_KATA_SMALL_KE ; c++ ) {
		char_order[c] = c+512;
		if (c == JIS_KATA_VU ) {
			char_order[c] = JIS_KATA_U + 512;
		}
	}
}


#ifdef	EUC
#define	KANA_BYTES	2
#else
#define	KANA_BYTES	2
#endif
  
static int
convert_htoz(unsigned char *str, int *ret)
{
	int c, c2;

	int num = KANA_BYTES;

#ifdef	EUC
	str++;
#endif
	c = *str++;
	if (!is201kana(*str)) {
		if (c >= 0xa0 && c <= 0xdf)
			*ret = kana[c - 0xa0];
		else *ret = 0x8e00 + c;
		return num;
	}
#ifdef	EUC
	str++;
#endif
	num += KANA_BYTES;
	
	if ( (c2 = *str++) == 0xde) {
		if      (c >= 0xb6 && c <= 0xba)                /* line-ga */
			c = daku[c - 0xb6];
		else if (c >= 0xbb && c <= 0xbf)                /* line-za */
			c = daku[c - 0xbb+5];
		else if (c >= 0xc0 && c <= 0xc4)                /* line-da */
			c = daku[c - 0xc0+10];
		else if (c >= 0xca && c <= 0xce)                /* line-ba */
			c = daku[c - 0xca+15];
		else if ( c == 0xb3 )                           /* vu */
			c = JIS_KATA_VU;
	} else if (c2 == 0xdf) {
		if (c >= 0xca && c <= 0xce)                     /* line-pa */
			c = handaku[c - 0xca];
	} else if ( c2 == 0xb1 ) {
		if (c >= 0xa0 && c <= 0xdd)
			*ret = kana[c - 0xa0];
		else *ret = 0x8e00 + c;
	}
	if ( c < 0x100 ) c += 0x8e00;
	return num;
}

int
isadddaku(int c)
{
	int	i;
	for ( i = 0 ; i < sizeof(dakuall) / sizeof(dakuall[1]) ; i++ )
		if ( dakuall[i]-1 == c ) return 1;
	return 0;
}

int
isaddhandaku(int c)
{
	int	i;
	for ( i = 0 ; i < sizeof(handakuall) / sizeof(handakuall[1]) ; i++ )
		if ( handakuall[i]-2 == c ) return 1;
	return 0;
}

int
isdaku(int c)
{
	int i;
	for ( i = 0 ; i < sizeof(dakuall) / sizeof(dakuall[1]) ; i++ )
		if ( dakuall[i] == c ) return 1;
	return 0;
}

int
ishandaku(int c)
{
	int i;
	for ( i = 0 ; i < sizeof(handakuall) / sizeof(handakuall[1]) ; i++ )
		if ( handakuall[i] == c ) return 1;
	return 0;
}

int mbchar(unsigned char *str, int *ret)
{
	int c = *str++;
	int cc, ccc;
	if ( !iskanji(c) && !is201kana(c)) {
		*ret = c & 0xff;
		return 1;
	}
	if ( iskanji (c) ) {
		cc = ((c & 0xff) << 8) | ((*str++) & 0xff);
		ccc = (((*str++ ) & 0xff) << 8 ) | ((*str++) & 0xff);

		if ( ccc == JIS_DAKUTEN ) {
			if ( isadddaku(cc)) cc++;
			else if ( cc == JIS_KATA_U || cc == JIS_HIRA_U )
				cc = JIS_KATA_VU;
			*ret = cc;
			return 4;
		} else if ( ccc == JIS_HANDAKUTEN ) {
			if ( isaddhandaku(cc)) cc+=2;
			*ret = cc;
			return 4;
		} else if ( ccc == JIS_CHOUON ) {
			*ret = cc;
			return 4;
		} else {
			*ret = cc;
			return 2;
		}
	} else if ( is201kana(c)) {
		/* EUC */
		return convert_htoz(str-1,ret);
	}
	return 1; /* never come here. Only to avoid a warning */
}

void
xinitial(unsigned char *str, int c)
{
	int up;
	if (isdaku(c)) c--;
	else if (ishandaku(c)) c -= 2;
	else if ( c == JIS_KATA_VU ) c = JIS_KATA_U;
#ifdef SJIS  /* A.K. 1999 */
	if (((up= (c & 0xff00) >>8) == JIS_KATA_UPPER) &&
            (c & 0xff) < 0x97)
#else
	if ((up= (c & 0xff00) >>8) == JIS_KATA_UPPER) 
#endif
                                                        { /* katakana */
		c += JIS_HIRA_SMALL_A - JIS_KATA_SMALL_A;
#ifdef	SJIS
		if ( (c & 0xff) > 0xdd ) c--;    /* A.K. 1999 */
#endif		
	}
	*str++ = (c >> 8 ) & 0xfff;
	*str++ = c & 0xff;
	*str= 0;
}

int
make_initial(unsigned char *src, unsigned char *dst)
{
	int len;
	int c;

	len = mbchar(src,&c);
	if ( len == 1 ) {
		dst[0] =toupper(c);
		dst[1] = 0;
	} else {
		xinitial(dst,c);
		len = 2;
	}
	return len;
}
#endif
void
init_char_order (void)
{
  int i;
  for (i = 1; i < 256; i++)
    char_order[i] = i;

  for (i = '0'; i <= '9'; i++)
    char_order[i] += 512;

  for (i = 'a'; i <= 'z'; i++)
    {
      char_order[i] = 512 + i;
      char_order[i + 'A' - 'a'] = 512 + i;
    }
#ifdef	KANJI  
	kana_char_order(); /* japanese */
#endif
}

/* Compare two fields (each specified as a start pointer and a character count)
   according to KEYFIELD.
   The sign of the value reports the relation between the fields. */

int
compare_field (struct keyfield *keyfield, char *start1, long int length1,
               long int pos1, char *start2, long int length2, long int pos2)
{
  if (keyfields->positional)
    {
      if (pos1 > pos2)
        return 1;
      else
        return -1;
    }
  if (keyfield->numeric)
    {
      long value = find_value (start1, length1) - find_value (start2, length2);
      if (value > 0)
        return 1;
      if (value < 0)
        return -1;
      return 0;
    }
  else
    {
      char *p1 = start1;
      char *p2 = start2;
      char *e1 = start1 + length1;
      char *e2 = start2 + length2;

      while (1)
        {
          int c1, c2;

          if (p1 == e1)
            c1 = 0;
          else
#ifdef	KANJI		
            p1 += mbchar(p1,&c1); /* japanese */
#else
            c1 = *p1++;
#endif
          if (p2 == e2)
            c2 = 0;
          else
#ifdef	KANJI		  
            p2 += mbchar(p2,&c2); /* japanese */
#else
            c2 = *p2++;
#endif	  

          if (char_order[c1] != char_order[c2])
            return char_order[c1] - char_order[c2];
          if (!c1)
            break;
        }

      /* Strings are equal except possibly for case.  */
      p1 = start1;
      p2 = start2;
      while (1)
        {
          int c1, c2;

          if (p1 == e1)
            c1 = 0;
          else
#ifdef	KANJI		  
	    p1 += mbchar(p1,&c1); /* japanese */
#else
            c1 = *p1++;
#endif
          if (p2 == e2)
            c2 = 0;
          else
#ifdef	KANJI		  
	    p2 += mbchar(p2,&c2); /* japanese */	  
#else
 	    c2 = *p2++;
#endif	  
#ifdef	KANJI
	  if ( iskanji(c1) || is201kana(c1)) {
	    if (c1 != c2)
	      /* Reverse sign here so upper case comes out last.  */
	      return c2 - c1;
	  } else {
	    if (c1 != c2)
	      return c1 - c2;
	  }
#else
 	  if (c1 != c2)
 	    /* Reverse sign here so upper case comes out last.  */
 	    return c2 - c1;
#endif
          if (!c1)
            break;
        }

      return 0;
    }
}

/* Sort INFILE, whose size is TOTAL,
   assuming that is small enough to be done in-core,
   then indexify it and send the output to OUTFILE (or to stdout).  */

#ifdef WIN32
#include "mkind.h"
#endif
void
sort_in_core (char *infile, int total, char *outfile)
{
  char **nextline;
  char *data = (char *) xmalloc (total + 1);
  char *file_data;
  long file_size;
  int i;
  FILE *ostream = stdout;
  struct lineinfo *lineinfo;

  /* Read the contents of the file into the moby array `data'. */

  int desc = open (infile, O_RDONLY, 0);

  if (desc < 0)
    fatal (_("failure reopening %s"), infile);
  for (file_size = 0;;)
    {
      i = read (desc, data + file_size, total - file_size);
      if (i <= 0)
        break;
      file_size += i;
    }
  file_data = data;
  data[file_size] = 0;

  close (desc);

  if (file_size > 0 && data[0] != '\\' && data[0] != '@')
    {
      error (_("%s: not a texinfo index file"), infile);
      return;
    }

  init_char_order ();

  /* Sort routines want to know this address. */

  text_base = data;

  /* Create the array of pointers to lines, with a default size
     frequently enough.  */

  nlines = total / 50;
  if (!nlines)
    nlines = 2;
  linearray = (char **) xmalloc (nlines * sizeof (char *));

  /* `nextline' points to the next free slot in this array.
     `nlines' is the allocated size.  */

  nextline = linearray;

  /* Parse the input file's data, and make entries for the lines.  */

  nextline = parsefile (infile, nextline, file_data, file_size);
  if (nextline == 0)
    {
      error (_("%s: not a texinfo index file"), infile);
      return;
    }

  /* Sort the lines. */

  /* If we have enough space, find the first keyfield of each line in advance.
     Make a `struct lineinfo' for each line, which records the keyfield
     as well as the line, and sort them.  */

  lineinfo = (struct lineinfo *) malloc ((nextline - linearray) * sizeof (struct lineinfo));

  if (lineinfo)
    {
      struct lineinfo *lp;
      char **p;

      for (lp = lineinfo, p = linearray; p != nextline; lp++, p++)
        {
          lp->text = *p;
          lp->key.text = find_field (keyfields, *p, &lp->keylen);
          if (keyfields->numeric)
            lp->key.number = find_value (lp->key.text, lp->keylen);
        }

      qsort ((char *)lineinfo, nextline - linearray, sizeof (struct lineinfo),
             compare_prepared);

      for (lp = lineinfo, p = linearray; p != nextline; lp++, p++)
        *p = lp->text;

      free (lineinfo);
    }
  else
    qsort ((char *)linearray, nextline - linearray, sizeof (char *), compare_full);

  /* Open the output file. */

  if (outfile)
    {
#ifdef WIN32
      ostream = fopen (outfile, "wb");
#else
      ostream = fopen (outfile, "w");
#endif
      if (!ostream)
        pfatal_with_name (outfile);
    }

  writelines (linearray, nextline - linearray, ostream);
  if (outfile)
    fclose (ostream);

  free (linearray);
  free (data);
}

/* Parse an input string in core into lines.
   DATA is the input string, and SIZE is its length.
   Data goes in LINEARRAY starting at NEXTLINE.
   The value returned is the first entry in LINEARRAY still unused.
   Value 0 means input file contents are invalid.  */

char **
parsefile (char *filename, char **nextline, char *data, long int size)
{
  char *p, *end;
  char **line = nextline;

  p = data;
  end = p + size;
  *end = 0;

  while (p != end)
    {
      if (p[0] != '\\' && p[0] != '@')
        return 0;

      *line = p;

      /* Find the first letter of the first field of this line.  If it
         is different from the first letter of the first field of the
         first line, we need initial headers in the output index.  */
      while (*p && *p != '{')
        p++;
      if (p == end)
        return 0;
      p++;
      if (first_initial)
        {
          if (first_initial != toupper (*p))
            need_initials = 1;
        }
      else
        first_initial = toupper (*p);

      while (*p && *p != '\n')
        p++;
      if (p != end)
        p++;

      line++;
      if (line == linearray + nlines)
        {
          char **old = linearray;
          linearray = (char **) xrealloc (linearray, sizeof (char *) * (nlines *= 4));
          line += linearray - old;
        }
    }

  return line;
}

/* Indexification is a filter applied to the sorted lines
   as they are being written to the output file.
   Multiple entries for the same name, with different page numbers,
   get combined into a single entry with multiple page numbers.
   The first braced field, which is used for sorting, is discarded.
   However, its first character is examined, folded to lower case,
   and if it is different from that in the previous line fed to us
   a \initial line is written with one argument, the new initial.

   If an entry has four braced fields, then the second and third
   constitute primary and secondary names.
   In this case, each change of primary name
   generates a \primary line which contains only the primary name,
   and in between these are \secondary lines which contain
   just a secondary name and page numbers. */

/* The last primary name we wrote a \primary entry for.
   If only one level of indexing is being done, this is the last name seen. */
char *lastprimary;
/* Length of storage allocated for lastprimary. */
int lastprimarylength;

/* Similar, for the secondary name. */
char *lastsecondary;
int lastsecondarylength;

/* Zero if we are not in the middle of writing an entry.
   One if we have written the beginning of an entry but have not
   yet written any page numbers into it.
   Greater than one if we have written the beginning of an entry
   plus at least one page number. */
int pending;

/* The initial (for sorting purposes) of the last primary entry written.
   When this changes, a \initial {c} line is written */

char *lastinitial;

int lastinitiallength;

/* When we need a string of length 1 for the value of lastinitial,
   store it here.  */

/*
char lastinitial1[2];
*/
char lastinitial1[3]; /* japanese */

/* Initialize static storage for writing an index. */

void
init_index (void)
{
  pending = 0;
  lastinitial = lastinitial1;
  lastinitial1[0] = 0;
  lastinitial1[1] = 0;
#ifdef KANJI /* 2003 --ak */
  lastinitial1[2] = 0;
#endif
  lastinitiallength = 0;
  lastprimarylength = 100;
  lastprimary = (char *) xmalloc (lastprimarylength + 1);
  memset (lastprimary, '\0', lastprimarylength + 1);
  lastsecondarylength = 100;
  lastsecondary = (char *) xmalloc (lastsecondarylength + 1);
  memset (lastsecondary, '\0', lastsecondarylength + 1);
}

/* Indexify.  Merge entries for the same name,
   insert headers for each initial character, etc.  */

void
indexify (char *line, FILE *ostream)
{
  char *primary, *secondary, *pagenumber;
  int primarylength, secondarylength = 0, pagelength;
  int nosecondary;
  int initiallength;
  char *initial;
/*
  char initial1[2];
*/
  char initial1[3]; /* japanese */
  register char *p;

  /* First, analyze the parts of the entry fed to us this time. */

  p = find_braced_pos (line, 0, 0, 0);
  if (*p == '{')
    {
      initial = p;
      /* Get length of inner pair of braces starting at `p',
         including that inner pair of braces.  */
      initiallength = find_braced_end (p + 1) + 1 - p;
    }
  else
    {
      initial = initial1;
#ifdef	KANJI
      initiallength = make_initial(p,initial1);	/* japanese */
#else	  
      initial1[0] = toupper (*p);
      initial1[1] = 0;
      initiallength = 1;
#endif
    }

  pagenumber = find_braced_pos (line, 1, 0, 0);
  pagelength = find_braced_end (pagenumber) - pagenumber;
  if (pagelength == 0)
    fatal (_("No page number in %s"), line);

  primary = find_braced_pos (line, 2, 0, 0);
  primarylength = find_braced_end (primary) - primary;

  secondary = find_braced_pos (line, 3, 0, 0);
  nosecondary = !*secondary;
  if (!nosecondary)
    secondarylength = find_braced_end (secondary) - secondary;

  /* If the primary is different from before, make a new primary entry. */
  if (strncmp (primary, lastprimary, primarylength))
    {
      /* Close off current secondary entry first, if one is open. */
      if (pending)
        {
          fputs ("}\n", ostream);
          pending = 0;
        }

      /* If this primary has a different initial, include an entry for
         the initial. */
      if (need_initials &&
          (initiallength != lastinitiallength ||
           strncmp (initial, lastinitial, initiallength)))
        {
          fprintf (ostream, "\\initial {");
          fwrite (initial, 1, initiallength, ostream);
          fputs ("}\n", ostream);
          if (initial == initial1)
            {
              lastinitial = lastinitial1;
              *lastinitial1 = *initial1;
#ifdef KANJI /* 2003 --ak */
              *(lastinitial1+1) = *(initial1+1);
#endif
            }
          else
            {
              lastinitial = initial;
            }
          lastinitiallength = initiallength;
        }

      /* Make the entry for the primary.  */
      if (nosecondary)
        fputs ("\\entry {", ostream);
      else
        fputs ("\\primary {", ostream);
      fwrite (primary, primarylength, 1, ostream);
      if (nosecondary)
        {
          fputs ("}{", ostream);
          pending = 1;
        }
      else
        fputs ("}\n", ostream);

      /* Record name of most recent primary. */
      if (lastprimarylength < primarylength)
        {
          lastprimarylength = primarylength + 100;
          lastprimary = (char *) xrealloc (lastprimary,
                                           1 + lastprimarylength);
        }
      strncpy (lastprimary, primary, primarylength);
      lastprimary[primarylength] = 0;

      /* There is no current secondary within this primary, now. */
      lastsecondary[0] = 0;
    }

  /* Should not have an entry with no subtopic following one with a
     subtopic. */

  if (nosecondary && *lastsecondary)
    error (_("entry %s follows an entry with a secondary name"), line);

  /* Start a new secondary entry if necessary. */
  if (!nosecondary && strncmp (secondary, lastsecondary, secondarylength))
    {
      if (pending)
        {
          fputs ("}\n", ostream);
          pending = 0;
        }

      /* Write the entry for the secondary.  */
      fputs ("\\secondary {", ostream);
      fwrite (secondary, secondarylength, 1, ostream);
      fputs ("}{", ostream);
      pending = 1;

      /* Record name of most recent secondary. */
      if (lastsecondarylength < secondarylength)
        {
          lastsecondarylength = secondarylength + 100;
          lastsecondary = (char *) xrealloc (lastsecondary,
                                             1 + lastsecondarylength);
        }
      strncpy (lastsecondary, secondary, secondarylength);
      lastsecondary[secondarylength] = 0;
    }

  /* Here to add one more page number to the current entry. */
  if (pending++ != 1)
    fputs (", ", ostream);  /* Punctuate first, if this is not the first. */
  fwrite (pagenumber, pagelength, 1, ostream);
}

/* Close out any unfinished output entry. */

void
finish_index (FILE *ostream)
{
  if (pending)
    fputs ("}\n", ostream);
  free (lastprimary);
  free (lastsecondary);
}

/* Copy the lines in the sorted order.
   Each line is copied out of the input file it was found in. */

void
writelines (char **linearray, int nlines, FILE *ostream)
{
  char **stop_line = linearray + nlines;
  char **next_line;

  init_index ();

  /* Output the text of the lines, and free the buffer space. */

  for (next_line = linearray; next_line != stop_line; next_line++)
    {
      /* Output the line only if distinct from previous one.  */
      if (next_line == linearray
      /* Compare previous line with this one, using only the
         explicitly specd keyfields. */
          || compare_general (*(next_line - 1), *next_line, 0L, 0L,
                              num_keyfields - 1))
        {
          char *p = *next_line;
          char c;

          while ((c = *p++) && c != '\n')
            /* Do nothing. */ ;
          *(p - 1) = 0;
          indexify (*next_line, ostream);
        }
    }

  finish_index (ostream);
}

/* Print error message and exit.  */

void
vdiag (const char *fmt, const char *diagtype, va_list ap)
{
  fprintf (stderr, "%s: ", program_name);
  if (diagtype)
    fprintf (stderr, "%s: ", diagtype);
  vfprintf (stderr, fmt, ap);
  putc ('\n', stderr);
}

void
error (const char *fmt, ...)
{
  va_list ap;

  va_start (ap, fmt);
  vdiag (fmt, NULL, ap);
  va_end (ap);
}

void
fatal (const char *fmt, ...)
{
  va_list ap;

  va_start (ap, fmt);
  vdiag (fmt, NULL, ap);
  va_end (ap);
  exit (EXIT_FAILURE);
}

void
perror_with_name (const char *name)
{
  fprintf (stderr, "%s: ", program_name);
  perror (name);
}

void
pfatal_with_name (const char *name)
{
  perror_with_name (name);
  exit (EXIT_FAILURE);
}


/* Return a newly-allocated string concatenating S1, S2, and S3.  */

static char *
concat3 (const char *s1, const char *s2, const char *s3)
{
  int len1 = strlen (s1), len2 = strlen (s2), len3 = strlen (s3);
  char *result = (char *) xmalloc (len1 + len2 + len3 + 1);

  strcpy (result, s1);
  strcpy (result + len1, s2);
  strcpy (result + len1 + len2, s3);
  *(result + len1 + len2 + len3) = 0;

  return result;
}
