/*
 * memalloc - allocate memory blocks from userspace
 *
 * Copyright (C) 2011 DSP Group
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the
 * Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA  02110-1301, USA.
*/
#define pr_fmt(fmt) KBUILD_MODNAME ": " fmt

#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/platform_device.h>
#include <linux/cdev.h>
#include <linux/slab.h>
#include <linux/ioctl.h>
#include <linux/fs.h>
#include <linux/mutex.h>
#include <linux/dma-mapping.h>
#include <linux/coherent_memalloc.h>
#include <asm/uaccess.h>
#include <linux/seq_file.h>
#include <linux/proc_fs.h>
#include <linux/mm_types.h>
#include <linux/mm.h>

MODULE_DESCRIPTION("contiguous physical memory allocator");
MODULE_AUTHOR("DSP Group");
MODULE_LICENSE("GPL");

/* on2 video hardware requires alignment to 4K, the code assumes the buffer alignment is a power of 2 */
#define BUFFER_ALIGNMENT    4096

struct memalloc_block {
    struct memalloc_block *prev;
    struct memalloc_block *next;
    
    phys_addr_t address;
    size_t size;
    struct file *owner;
};

struct simple_allocator {
    struct mutex lock;
    struct memalloc_block *free_list;
    struct memalloc_block *allocated_list;
};

struct memalloc {
	struct cdev cdev;
	struct platform_device *pdev;

    struct simple_allocator allocator;
};

static dev_t chrdev;
static struct memalloc *memalloc[1];

static int simple_allocator_init(struct simple_allocator *inst, phys_addr_t address, size_t size);
static int simple_free_user(struct simple_allocator *inst, struct file *owner);
static int simple_free(struct simple_allocator *inst, phys_addr_t address);
static phys_addr_t simple_allocate(struct simple_allocator *inst, size_t size, struct file *owner);
static int simple_verify_block(struct simple_allocator *inst, phys_addr_t address, size_t size, struct file *owner);
static void create_memalloc_proc_entry(void);

static int
memalloc_open(struct inode *inode, struct file *filp)
{
	unsigned long id = iminor(inode) - MINOR(chrdev);
	struct memalloc *a = memalloc[id];

	if (!a)
		return -ENODEV;

	filp->private_data = a;
	return 0;
}

static long
memalloc_ioctl(struct file *filp, unsigned int cmd, unsigned long arg)
{
	struct memalloc *memdev = filp->private_data;
	int ret = 0;

	if (arg == 0)
		return -EFAULT;

	/*
	 * extract the type and number bitfields, and don't decode
	 * wrong cmds: return ENOTTY (inappropriate ioctl) before access_ok()
	*/
	if(_IOC_TYPE(cmd) != MEMALLOC_IOC_MAGIC)
		return -ENOTTY;
	if(_IOC_NR(cmd) > MEMALLOC_IOC_MAXNR)
		return -ENOTTY;

	if(_IOC_DIR(cmd) & _IOC_READ)
		ret = !access_ok(VERIFY_WRITE, (void *) arg, _IOC_SIZE(cmd));
	else if(_IOC_DIR(cmd) & _IOC_WRITE)
		ret = !access_ok(VERIFY_READ, (void *) arg, _IOC_SIZE(cmd));
	if(ret)
		return -EFAULT;

	switch (cmd) 
    {
        case MEMALLOC_IOCHARDRESET:
            dev_err(&memdev->pdev->dev, "error: hardreset not allowed\n");
            ret = -EPERM;
            break;

	    case MEMALLOC_IOCXGETBUFFER:
            {
                MemallocParams memparams;

                ret = __copy_from_user(&memparams, (const void *) arg, sizeof(memparams));
                if (ret < 0) {
                    dev_err(&memdev->pdev->dev, "copy_from_user failed\n");
                    return ret;
                }

                dev_dbg(&memdev->pdev->dev, "getbuffer:  %p, size=%u\n", filp,
                        memparams.size);

                memparams.busAddress = simple_allocate(&memdev->allocator, memparams.size, filp);
                if (!memparams.busAddress) {
                    dev_err(&memdev->pdev->dev, "getbuffer: allocation failed (increase VPU memory?)\n");
                    return -ENOMEM;
                }

                dev_dbg(&memdev->pdev->dev, "getbuffer:  %p, pa=%#x\n", filp,
                        memparams.busAddress);

                ret = __copy_to_user((void *) arg, &memparams, sizeof(memparams));
                if (ret < 0)
                    dev_err(&memdev->pdev->dev, "copy_to_user failed\n");

                return ret;
            }

	    case MEMALLOC_IOCSFREEBUFFER:
            {
                unsigned long busaddr;

                __get_user(busaddr, (unsigned long *) arg);

                dev_dbg(&memdev->pdev->dev, "freebuffer: %p, pa=%#lx\n", filp,
                        busaddr);

                ret = simple_free(&memdev->allocator, busaddr);
                if (ret < 0) {
                    ret = -EFAULT;
                }

                return ret;
            }
	}

	return 0;
}

static int memalloc_mmap(struct file *filp, struct vm_area_struct *vma)
{
	struct memalloc *memdev = filp->private_data;

	size_t size = vma->vm_end - vma->vm_start;
	phys_addr_t address = vma->vm_pgoff << PAGE_SHIFT;
	
	if (simple_verify_block(&memdev->allocator, address, size, filp)) 
		return -EINVAL; // this block is not available for that user
	
	vma->vm_page_prot = phys_mem_access_prot(filp, vma->vm_pgoff,
						 size,
						 vma->vm_page_prot);

	if (remap_pfn_range(vma,
			    vma->vm_start,
			    vma->vm_pgoff,
			    size,
			    vma->vm_page_prot)) {
		return -EAGAIN;
	}

	return 0;
}

static int
memalloc_release(struct inode *inode, struct file *filp)
{
	struct memalloc *memdev = filp->private_data;

    simple_free_user(&memdev->allocator, filp);

	return 0;
}

struct file_operations memalloc_fops = {
	.open = memalloc_open,
	.unlocked_ioctl = memalloc_ioctl,
	.release = memalloc_release,
	.mmap = memalloc_mmap,
};

static int __init memalloc_probe(struct platform_device *pdev)
{
	struct memalloc *memdev;
	struct resource *r;
	phys_addr_t base;
	size_t size;
	int ret = -ENOMEM;
    unsigned int align_diff;
    
	r = platform_get_resource(pdev, IORESOURCE_MEM, 0);
	if (!r) {
		dev_err(&pdev->dev, "no memory resource given\n");
		return -EINVAL;
	}

	base = r->start;
	size = resource_size(r);

    /* check that base is aligned to 4K
       all buffers should be aligned to 4K
       */
    align_diff = ((base&(BUFFER_ALIGNMENT-1))>0)? BUFFER_ALIGNMENT - (base & (BUFFER_ALIGNMENT-1)) : 0;
    base += align_diff;
    size -= align_diff;
    size &= ~(BUFFER_ALIGNMENT-1);

	memdev = kzalloc(sizeof(struct memalloc), GFP_KERNEL);
	if (!memdev)
		return -ENOMEM;

	printk("%s(): address = %#x, size = %d bytes\n", __func__, base, size);

    ret = simple_allocator_init(&memdev->allocator, base, size);
    if (ret < 0) {
        dev_err(&pdev->dev, "simple allocator initialization failed!\n");
        goto out_free;
    }

	memdev->pdev = pdev;

	memalloc[0] = memdev;

	cdev_init(&memdev->cdev, &memalloc_fops);
	memdev->cdev.owner = THIS_MODULE;

	/* FIXME: handle multiple devices (device id) */
	ret = cdev_add(&memdev->cdev, MKDEV(MAJOR(chrdev), MINOR(chrdev)), 1);
	if (ret)
		dev_err(&pdev->dev, "error: cannot register chrdev\n");

   	/* Create proc entry */
    create_memalloc_proc_entry();

	return 0;

out_free:
	kfree(memdev);
	return ret;
}

static struct platform_driver memalloc_driver = {
	.driver = {
		.name = "memalloc",
		.owner = THIS_MODULE,
	},
};

static int __init memalloc_init(void)
{
	int ret;

	ret = alloc_chrdev_region(&chrdev,  0, 1, "memalloc");
	if (ret < 0) {
		pr_err("cannot get major\n");
		return ret;
	}

	pr_info("registered (major %d,  minor %d)\n", MAJOR(chrdev),
		MINOR(chrdev));

	return platform_driver_probe(&memalloc_driver, memalloc_probe);
}

static void memalloc_exit(void)
{
	unregister_chrdev(MAJOR(chrdev), "memalloc");
	platform_driver_unregister(&memalloc_driver);
}

module_init(memalloc_init)
module_exit(memalloc_exit)


/**************************************************************************************/
/* /proc/memalloc for debugging                                                       */
/**************************************************************************************/

static void list_print(struct seq_file *f, struct memalloc_block *head, char *title)
{
    size_t total_mem = 0;

    seq_printf(f, "%s\n", title);
    seq_printf(f, "| Address  | | Size     | | User   | - ( prev    <--   node   --> next    )\n");
    while (head != NULL) {
        seq_printf(f, "[0x%08x] [0x%08x] [%p] - (%p <-- %p --> %p)\n", head->address, head->size, head->owner, head->prev, head, head->next);
        total_mem += head->size;
        head = head->next;
    }
    seq_printf(f, "\n");
}

static int memalloc_status_show(struct seq_file *f, void *unused)
{
    /* Fix me: blindly refering to memalloc[0] */
    struct memalloc_block *free = memalloc[0]->allocator.free_list;
    struct memalloc_block *allocated = memalloc[0]->allocator.allocated_list;
    struct memalloc_block *temp;
    unsigned int total_free=0, total_allocated=0, num_blocks=0;


    mutex_lock(&(memalloc[0]->allocator.lock));

    list_print(f, free, "free list:");
    list_print(f, allocated, "allocated list:");

    seq_printf(f, "pool status:\n");
    seq_printf(f, "================\n");
    seq_printf(f, "* '-' <= 32KB free\n");
    seq_printf(f, "* 'x' <= 32KB allocated\n\n");


    if (free == NULL) {
        temp = allocated;
    }
    else if (allocated == NULL) {
        temp = free;
    } 
    else if (free->address < allocated->address) {
        temp = free;
    } 
    else {
        temp = allocated;
    }

    seq_printf(f, "[");
    while (temp != NULL) 
    {
        int spaces, i;
        char sign;
        if (temp->size < 32*1024)
            spaces = 1;
        else
            spaces = temp->size / (32*1024);

        if (temp == free) {
            sign = '-';
            total_free += temp->size;
        } else {
            sign = 'x';
            total_allocated += temp->size;
        }

        if (num_blocks++ > 0) seq_printf(f, "|");
        for (i=0 ; i < spaces ; i++) {
            seq_printf(f, "%c", sign);
        }

        if (temp == free) 
            free = free->next;
        else
            allocated = allocated->next;

        if (free == NULL) {
            temp = allocated;
        }
        else if (allocated == NULL) {
            temp = free;
        } else if (free->address < allocated->address) {
            temp = free;
        } else {
            temp = allocated;
        }
    }

    seq_printf(f, "]\n");
    seq_printf(f, "blocks = %d\n", num_blocks);
    seq_printf(f, "allocated = %d bytes\n", total_allocated);
    seq_printf(f, "free = %d bytes\n", total_free);
    seq_printf(f, "total = %d bytes\n\n", total_allocated + total_free);
    
    mutex_unlock(&(memalloc[0]->allocator.lock));

	return 0;
}

static int memalloc_status_open(struct inode *inode, struct file *file)
{
	return single_open(file, memalloc_status_show, NULL);
}

static const struct file_operations memalloc_status_fops = {
	.owner = THIS_MODULE,
	.open = memalloc_status_open,
	.read = seq_read,
	.llseek = seq_lseek,
	.release = single_release,
};

static void create_memalloc_proc_entry(void)
{
   	/* Create proc entry */
	proc_create_data("memalloc", S_IRUGO, NULL, &memalloc_status_fops, NULL);
}

/**************************************************************************************/
/* Simple dynamic allocator code below                                                */
/**************************************************************************************/

static void sorted_list_add(struct memalloc_block **head, struct memalloc_block *newblock)
{
    newblock->next = NULL;
    newblock->prev = NULL;

    if (*head == NULL) {
        *head = newblock;
    }
    else if ((*head)->address > newblock->address) {
        newblock->next = *head;
        newblock->prev = NULL;
        (*head)->prev = newblock;
        *head = newblock;
    }
    else {
        struct memalloc_block *temp = *head;
        while(temp->next != NULL) {
            if (temp->next->address > newblock->address) {
                temp->next->prev = newblock;
                break;
            }
            temp = temp->next;
        }

        // put the new block after temp
        newblock->next = temp->next;
        newblock->prev = temp;
        temp->next = newblock;
    }
}

static void sorted_list_remove(struct memalloc_block **head, struct memalloc_block *block)
{
    if (block == *head) {
        *head = block->next;
    }
    else {
        block->prev->next = block->next;
    }

    if (block->next != NULL) {
        block->next->prev = block->prev;
    }
}

static int simple_allocator_init(struct simple_allocator *inst, phys_addr_t address, size_t size)
{
    struct memalloc_block *block;

    inst->free_list = NULL;
    inst->allocated_list = NULL;

    mutex_init(&(inst->lock));

    /* init one big memory block */
    block = (struct memalloc_block*)kmalloc(sizeof(struct memalloc_block), GFP_KERNEL);
    if (!block) {
        return -ENOMEM;
    }

    block->address = address;
    block->size = size;
    block->owner = NULL;

    sorted_list_add(&(inst->free_list), block);
    return 0;
}


static phys_addr_t simple_allocate(struct simple_allocator *inst, size_t size, struct file *owner)
{
    struct memalloc_block *temp;

    if (!size || !owner) return 0;

    mutex_lock(&inst->lock);

    // always align to BUFFER_ALIGNMENT
    size = (size+BUFFER_ALIGNMENT-1) & ~(BUFFER_ALIGNMENT-1);

    /* look for available memory block */
    temp = inst->free_list;
    while (temp != NULL) {
        if (temp->size >= size) {
            // found
            struct memalloc_block *newblock = (struct memalloc_block*)kmalloc(sizeof(struct memalloc_block), GFP_KERNEL);
            if (!newblock) {
                return 0;
            }
            newblock->address = temp->address;
            newblock->size = size;
            newblock->owner = owner;
            sorted_list_add(&(inst->allocated_list), newblock);

            if (size == temp->size) {
                // no space left in this block, remove it from the list
                sorted_list_remove(&(inst->free_list), temp);
                kfree(temp);
            }
            else {
                // reduce free block size
                temp->address += size;
                temp->size -= size;
            }

            mutex_unlock(&inst->lock);
            return newblock->address;
        }
        temp = temp->next;
    }

    mutex_unlock(&inst->lock);
    return 0;
}

static int simple_free_block(struct simple_allocator *inst, struct memalloc_block *block)
{
    if (block == NULL) 
        return -1;

    block->owner = NULL;

    // remove from allocated_list and add to free_list
    sorted_list_remove(&(inst->allocated_list), block);
    sorted_list_add(&(inst->free_list), block);

    // try to merge its neighboring blocks in free_list
    if (block->next != NULL) {
        struct memalloc_block *next_block = block->next;

        if (block->address + block->size == next_block->address) {
            // we can merge the next block
            block->size += next_block->size;
            sorted_list_remove(&(inst->free_list), next_block);
            kfree(next_block);
        }
    }

    if (block->prev != NULL) {
        struct memalloc_block *prev_block = block->prev;

        if (prev_block->address + prev_block->size == block->address) {
            // we can merge the previous block
            block->address = prev_block->address;
            block->size += prev_block->size;
            sorted_list_remove(&(inst->free_list), prev_block);
            kfree(prev_block);
        }
    }

    return 0;
}

static int simple_free(struct simple_allocator *inst, phys_addr_t address)
{
    struct memalloc_block *temp;

    mutex_lock(&inst->lock);

    /* look for this address in allocated list */
    temp = inst->allocated_list;

    while(temp != NULL) {
        if (temp->address == address) {
            // found block
            
            int ret = simple_free_block(inst, temp);

            mutex_unlock(&inst->lock);
            return ret;
        }
        else temp = temp->next;
    }

    mutex_unlock(&inst->lock);
    return -1;
}

static int simple_free_user(struct simple_allocator *inst, struct file *owner)
{
    struct memalloc_block *temp;

    mutex_lock(&inst->lock);

    /* look for user in the allocated list */
    temp = inst->allocated_list;

    while(temp != NULL) {
        if (temp->owner == owner) {
            // found
            struct memalloc_block *next_allocated_block = temp->next;

            simple_free_block(inst, temp);

            temp = next_allocated_block;
        }
        else 
            temp = temp->next;
    }

    mutex_unlock(&inst->lock);
    return 0;
}

static int simple_verify_block(struct simple_allocator *inst, phys_addr_t address, size_t size, struct file *owner)
{
	struct memalloc_block *temp;

	mutex_lock(&inst->lock);

	/* look for user in the allocated list */
	temp = inst->allocated_list;

	while(temp != NULL) {
	    if (temp->owner == owner &&
		temp->address == address &&
		temp->size >= size) {
			// found
			mutex_unlock(&inst->lock);
			return 0;
	    }
	    else 
			temp = temp->next;
	}

	// not found
	mutex_unlock(&inst->lock);
	return -1;
}

