/*
 *  linux/arch/arm/mach-dmw/devices.c
 *
 *  Copyright (C) 2011 DSPG Technologies GmbH
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
 

#include <linux/fs.h>
#include <linux/file.h>
#include <linux/android_pmem.h>
#include <linux/dmw96dma.h>
#include <linux/init.h>
#include <linux/platform_device.h>
#include <linux/dma-mapping.h>
#include <linux/memblock.h>
#include <linux/dmw-gpio.h>

#include <mach/clock.h>
#include <mach/irqs.h>
#include <mach/powerdomain.h>
#include <mach/reset.h>
#include <mach/watchdog.h>
#include <asm/pmu.h>

#include "devices.h"
#include "arch.h"

struct platform_device video_enc_dev_pwrdom = {
	.name		= "dmw-powerdomain",
	.id		= 0,
	.dev = {
		.platform_data = &(struct dmw_pwrdom_pdata) {
			.domain	= DMW_PWRDOM_VIDEO_ENC_DEC,
			.name	= "VIDEO_ENC_DEC",
		},
	},
};

static struct dmw96dma_ch_config gdmac_configs[] = {
	/* ch08: TDM1 RX FIFO */
	[8] = {
		.direction       = 0, /* A -> B */
		.hw_port         = 0, /* Peripheral on Port A */
		.transfer_length = 2*16 -1, /* 16 audio frames (stereo samples) */
		.width           = DMW96DMA_WIDTH_32,
		.mode            = DMW96DMA_MODE_HW_SINGLE,
		.base            = DMW_TDM1_BASE + 0x3c,
	},
	/* ch09: TDM1 TX FIFO */
	[9] = {
		.direction       = 1, /* B -> A */
		.hw_port         = 0, /* Peripheral on Port A */
		.transfer_length = 2*16 -1, /* 16 audio frames (stereo samples) */
		.width           = DMW96DMA_WIDTH_32,
		.mode            = DMW96DMA_MODE_HW_SINGLE,
		.base            = DMW_TDM1_BASE + 0x38,
	},
	/* ch10: TDM2 RX FIFO */
	[10] = {
		.direction       = 0, /* A -> B */
		.hw_port         = 0, /* Peripheral on Port A */
		.transfer_length = 2*16 -1, /* 16 audio frames (stereo samples) */
		.width           = DMW96DMA_WIDTH_32,
		.mode            = DMW96DMA_MODE_HW_SINGLE,
		.base            = DMW_TDM2_BASE + 0x3c,
	},
	/* ch11: TDM2 TX FIFO */
	[11] = {
		.direction       = 1, /* B -> A */
		.hw_port         = 0, /* Peripheral on Port A */
		.transfer_length = 2*16 -1, /* 16 audio frames (stereo samples) */
		.width           = DMW96DMA_WIDTH_32,
		.mode            = DMW96DMA_MODE_HW_SINGLE,
		.base            = DMW_TDM2_BASE + 0x38,
	},
	/* ch12: TDM3 RX FIFO */
	[12] = {
		.direction       = 0, /* A -> B */
		.hw_port         = 0, /* Peripheral on Port A */
		.transfer_length = 2*16 -1, /* 16 audio frames (stereo samples) */
		.width           = DMW96DMA_WIDTH_32,
		.mode            = DMW96DMA_MODE_HW_SINGLE,
		.base            = DMW_TDM3_BASE + 0x3c,
	},
	/* ch13: TDM3 TX FIFO */
	[13] = {
		.direction       = 1, /* B -> A */
		.hw_port         = 0, /* Peripheral on Port A */
		.transfer_length = 2*16 -1, /* 16 audio frames (stereo samples) */
		.width           = DMW96DMA_WIDTH_32,
		.mode            = DMW96DMA_MODE_HW_SINGLE,
		.base            = DMW_TDM3_BASE + 0x38,
	},
	/* ch14: SPI1 RX FIFO */
	[14] = {
		.direction       = 0, /* A -> B */
		.hw_port         = 0, /* Peripheral on Port A */
		.transfer_length = 4 -1, /* 16 bytes or 4 words */
		.width           = DMW96DMA_WIDTH_8,
		.mode            = DMW96DMA_MODE_HW_SINGLE,
		.base            = DMW_SPI1_BASE + 0x1C,
	},
	/* ch15: SPI1 TX FIFO */
	[15] = {
		.direction       = 1, /* B -> A */
		.hw_port         = 0, /* Peripheral on Port A */
		.transfer_length = 4 -1, /* n+1: 16 bytes or 4 words */
		.width           = DMW96DMA_WIDTH_8,
		.mode            = DMW96DMA_MODE_HW_SINGLE,
		.base            = DMW_SPI1_BASE + 0x18,
	},
	/* ch16: SPI2 RX FIFO */
	[16] = {
		.direction       = 0, /* A -> B */
		.hw_port         = 0, /* Peripheral on Port A */
		.transfer_length = 4 -1, /* n+1: 16 bytes or 4 words */
		.width           = DMW96DMA_WIDTH_8,
		.mode            = DMW96DMA_MODE_HW_SINGLE,
		.base            = DMW_SPI2_BASE + 0x1C,
	},
	/* ch17: SPI2 TX FIFO */
	[17] = {
		.direction       = 1, /* B -> A */
		.hw_port         = 0, /* Peripheral on Port A */
		.transfer_length = 4 -1, /* n+1: 16 bytes or 4 words */
		.width           = DMW96DMA_WIDTH_8,
		.mode            = DMW96DMA_MODE_HW_SINGLE,
		.base            = DMW_SPI2_BASE + 0x18,
	},
};

/*
 * DMA channel priorities. Valid values are from 0 to 31 where 0 has the
 * highest priority. Every priority must be used only once, including the ones
 * given below for rotate and memcpy!
 */
u8 gdmac_priorities[ARRAY_SIZE(gdmac_configs)] = {
	 0, /* ch00: PACP IN FIFO */
	 1, /* ch01: PACP OUT FIFO */
	 2, /* ch02: UART1 RX FIFO */
	 3, /* ch03: UART1 TX FIFO */
	 4, /* ch04: UART2 RX FIFO */
	 5, /* ch05: UART2 TX FIFO */
	 6, /* ch06: UART3 RX FIFO */
	 7, /* ch07: UART3 TX FIFO */
	 8, /* ch08: TDM1 RX FIFO */
	 9, /* ch09: TDM1 TX FIFO */
	10, /* ch10: TDM2 RX FIFO */
	11, /* ch11: TDM2 TX FIFO */
	12, /* ch12: TDM3 RX FIFO */
	13, /* ch13: TDM3 TX FIFO */
	14, /* ch14: SPI2 RX FIFO */
	15, /* ch15: SPI2 TX FIFO */
	16, /* ch16: SPI1 RX FIFO */
	17, /* ch17: SPI1 TX FIFO */
};

static struct dmw96dma_platform_data gdmac_pdata = {
	.timeout         = 0x0FFF,
	.desc_port       = 1,
	.num_channels    = ARRAY_SIZE(gdmac_configs),
	.ch_configs      = gdmac_configs,
	.ch_priorities   = gdmac_priorities,
	.rotate_priority = 18,
	.memcpy_priority = 19, /* uses priorities 19 and 20! */
	.memcpy_base     = DMW_SRAM_GDMAC_BASE,
};

static struct resource gdmac_resources[] = {
	{
		.start	= DMW_IID_GDMAC_DMA_DONE,
		.end	= DMW_IID_GDMAC_DMA_DONE,
		.flags	= IORESOURCE_IRQ,
	},
	{
		.start	= DMW_IID_GDMAC_ERROR,
		.end	= DMW_IID_GDMAC_ERROR,
		.flags	= IORESOURCE_IRQ,
	},
	{
		.start	= DMW_GDMAC_BASE,
		.end	= DMW_GDMAC_BASE + 0x200 - 1,
		.flags	= IORESOURCE_MEM,
	},
};

static struct platform_device gdmac_device = {
	.name		= "dmw96dma",
	.id		= 0,
	.dev		= {
		.coherent_dma_mask = ~0,
		.platform_data     = &gdmac_pdata,
	},
	.num_resources	= ARRAY_SIZE(gdmac_resources),
	.resource	= gdmac_resources,
};

static struct resource dmw96_osdm_resources[] = {
	{
		.start = DMW_OSDM_BASE,
		.end   = DMW_OSDM_BASE + SZ_4K - 1,
		.flags = IORESOURCE_MEM,
	},
	{
		.start = DMW_IID_OSDM,
		.flags = IORESOURCE_IRQ,
		.name  = "dmw96osdm-irq",
	},
};

static struct platform_device osdm_device = {
	.name = "dmw96osdm",
	.dev		= {
		.coherent_dma_mask = DMA_BIT_MASK(32),
	},
	.num_resources = ARRAY_SIZE(dmw96_osdm_resources),
	.resource = dmw96_osdm_resources,
};

static struct resource gpu_resources[] = {
	{
		.name	= "gpu_irq",
		.start	= DMW_IID_GPU,
		.flags	= IORESOURCE_IRQ,
	},
	{
		.name	= "gpu_base",
		.start	= DMW_GPU_BASE,
		.end	= DMW_GPU_BASE + SZ_4K - 1,
		.flags	= IORESOURCE_MEM,
	},
	{
		/* updated in arch_initcall */
		.name	= "gpu_mem",
		.flags	= IORESOURCE_DMA,
	},
};

static struct platform_device gpu_device = {
	.name		= "galcore",
	.num_resources	= ARRAY_SIZE(gpu_resources),
	.resource	= gpu_resources,
	.dev.coherent_dma_mask = ~0,
};

static struct resource sec_resources[] = {
	{
		.start	= DMW_IID_SEC,
		.end	= DMW_IID_SEC,
		.flags	= IORESOURCE_IRQ,
	},
	{
		.start	= DMW_IID_SW1,
		.end	= DMW_IID_SW1,
		.flags	= IORESOURCE_IRQ,
	},
	{
		.start	= DMW_SEC_BASE,
		.end	= DMW_SEC_BASE + 0x1000 - 1,
		.flags	= IORESOURCE_MEM,
	},
};

static struct platform_device sec_device = {
	.name		= "sec",
	.num_resources	= ARRAY_SIZE(sec_resources),
	.resource	= sec_resources,
};

static struct resource dbm_resources[] = {
	{
		.start	= DMW_IID_DBM,
		.end	= DMW_IID_DBM,
		.flags	= IORESOURCE_IRQ,
	},
	{
		.start	= DMW_DEBUG_BASE,
		.end	= DMW_DEBUG_BASE + 0x1000 - 1,
		.flags	= IORESOURCE_MEM,
	},
};

static struct platform_device dbm_device = {
	.name		= "dbm",
	.num_resources	= ARRAY_SIZE(dbm_resources),
	.resource	= dbm_resources,
};

static struct dmw_wdt_platform_data watchdog_pdata = {
	.get_wdt_reset_ind = dmw_get_wdt_reset_ind,
};

static struct resource watchdog_resources[] = {
	{
		.start	= DMW_IID_WD,
		.end	= DMW_IID_WD,
		.flags	= IORESOURCE_IRQ,
	},
	{
		.start	= DMW_WATCHDOG_BASE,
		.end	= DMW_WATCHDOG_BASE + SZ_4K - 1,
		.flags	= IORESOURCE_MEM,
	},
};

static struct platform_device watchdog_device = {
	.name		= "dmw_wdt",
	.id		= -1,
	.dev		= {
		.platform_data = &watchdog_pdata,
	},
	.num_resources	= ARRAY_SIZE(watchdog_resources),
	.resource	= watchdog_resources,
};

static struct resource memalloc_resources[] = {
	{
		/* updated in arch_initcall */
		.flags = IORESOURCE_MEM,
	},
};

static struct platform_device memalloc_device = {
	.name = "memalloc",
	.resource = memalloc_resources,
	.num_resources = ARRAY_SIZE(memalloc_resources),
};

static struct resource hx280enc_resources[] = {
	{
		.start	= DMW_VPU_ENC_BASE,
		.end	= DMW_VPU_ENC_BASE + SZ_4K - 1,
		.flags	= IORESOURCE_MEM,
	},
	{
		.start	= DMW_IID_VIDEO_ENCODER,
		.end	= DMW_IID_VIDEO_ENCODER,
		.flags	= IORESOURCE_IRQ,
	},
};

static struct platform_device hx280enc_device = {
	.name		= "hx280enc",
	.num_resources	= ARRAY_SIZE(hx280enc_resources),
	.resource	= hx280enc_resources,
	.dev		 = {
		.parent = &video_enc_dev_pwrdom.dev,
	},
};

static struct resource hx170dec_resources[] = {
	{
		.start	= DMW_VPU_DEC_BASE,
		.end	= DMW_VPU_DEC_BASE + SZ_4K - 1,
		.flags	= IORESOURCE_MEM,
	},
	{
		.start	= DMW_IID_VIDEO_DECODER,
		.end	= DMW_IID_VIDEO_DECODER,
		.flags	= IORESOURCE_IRQ,
	},
};

static struct platform_device hx170dec_device = {
	.name		= "hx170dec",
	.num_resources	= ARRAY_SIZE(hx170dec_resources),
	.resource	= hx170dec_resources,
	.dev		 = {
		.parent = &video_enc_dev_pwrdom.dev,
	},
};

static struct resource dmw_pmu_resources[] = {
	[0] = {
		.start  = DMW_IID_CORTEX,
		.end    = DMW_IID_CORTEX,
		.flags  = IORESOURCE_IRQ,
	},
};

static struct platform_device dmw_pmu_device = {
	.name           = "arm-pmu",
	.id             = ARM_PMU_DEVICE_CPU,
	.num_resources  = ARRAY_SIZE(dmw_pmu_resources),
	.resource       = dmw_pmu_resources,
};

static struct resource dmw_gpio_resources[] = {
	{
		.start	= DMW_GPIO_BASE,
		.end	= DMW_GPIO_BASE + (0x60*7) - 1,
		.flags	= IORESOURCE_MEM,
	},
};

static u32 dmw_gpio_nx_masks[7] = {
	[4] = 0x7FFC0000, /* EGPIO18..EGPIO30 */
	[6] = 0xFF800000, /* GGPIO23..GGPIO31 */
};

static struct dmw_gpio_pdata dmw_gpio_pdata = {
	.num_banks = 7,
	.nx_masks = dmw_gpio_nx_masks,
};

static struct platform_device dmw_gpio_device = {
	.name		= "dmw-gpio",
	.id		= -1,
	.dev		= {
		.platform_data     = &dmw_gpio_pdata,
	},
	.num_resources  = ARRAY_SIZE(dmw_gpio_resources),
	.resource       = dmw_gpio_resources,
};

static struct platform_device *dmw_platform_devs[] __initdata = {
	&video_enc_dev_pwrdom,
	&dbm_device,
	&gdmac_device,
	&hx170dec_device,
	&hx280enc_device,
	&osdm_device,
	&sec_device,
	&watchdog_device,
	&dmw_pmu_device,
	&dmw_gpio_device,
};

int dmw_dev_reserve_dmamem(struct device *dev, size_t size, int exclusive)
{
	int flags = DMA_MEMORY_MAP;
	phys_addr_t pa;
	int ret;

	if (exclusive)
		flags |= DMA_MEMORY_EXCLUSIVE;

	ret = dmw_reservemem_get(&pa, size);
	if (ret)
		return -ENOMEM;

	ret = dma_declare_coherent_memory(dev, pa, pa, size, flags);
	if(!(ret & DMA_MEMORY_MAP))
		return -ENOMEM;

	return 0;
}

static int __init dmw_arch_dev_init(void)
{
	struct resource *r;
	int ret;

	/* update gpu resource with reserved system memory and register */
	r = platform_get_resource(&gpu_device, IORESOURCE_DMA, 0);
	ret = dmw_reservemem_get(&r->start, DMW_RESERVEMEM_GPU);
	if (!ret) {
		r->end = r->start + DMW_RESERVEMEM_GPU - 1;
		platform_device_register(&gpu_device);
	}

	/* update memalloc resource with reserved system memory */
	r = platform_get_resource(&memalloc_device, IORESOURCE_MEM, 0);
	ret = dmw_reservemem_get(&r->start, DMW_RESERVEMEM_MEMALLOC);
	if (!ret) {
		r->end = r->start + DMW_RESERVEMEM_MEMALLOC - 1;
		platform_device_register(&memalloc_device);
	}

	return platform_add_devices(dmw_platform_devs, ARRAY_SIZE(dmw_platform_devs));
}

arch_initcall(dmw_arch_dev_init);

