/*
 * arch/arm/mach-dmw/sched_stats.c
 *
 * Copyright (C) 2012 DSP Group
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/vmalloc.h>
#include <linux/sched.h>
#include <linux/slab.h>
#include <linux/platform_device.h>
#include <linux/debugfs.h>
#include <asm/thread_notify.h>

static int count = 100000;
module_param(count, int, 0644);
MODULE_PARM_DESC(count, "Number of entries for scheduler statistics.");

struct sched_stat {
	struct timespec start;
	struct timespec end;
	char name[TASK_COMM_LEN];
	int pid;
};
static struct sched_stat *stats;
static int cur_stat = 0, next_stat = 1;
static DEFINE_SPINLOCK(dmw_sched_stats_lock);

static void
dmw_sched_stats_add(struct task_struct *curr, struct task_struct *next)
{
	unsigned long flags;

	spin_lock_irqsave(&dmw_sched_stats_lock, flags);
	getnstimeofday(&stats[cur_stat].end);

	cur_stat = next_stat;
	next_stat++;
	if (next_stat >= count)
		next_stat = 0;

	getnstimeofday(&stats[cur_stat].start);
	memcpy(stats[cur_stat].name, next->comm, TASK_COMM_LEN);
	stats[cur_stat].pid = next->pid;

	stats[next_stat].name[0] = 0; /* start/end marker */
	spin_unlock_irqrestore(&dmw_sched_stats_lock, flags);
}

static int
dmw_sched_stats_event(struct notifier_block *self, unsigned long cmd, void *t)
{
	struct thread_info *thread = (struct thread_info *)t;

	switch (cmd) {
	case THREAD_NOTIFY_SWITCH:
		dmw_sched_stats_add(current_thread_info()->task, thread->task);
		break;
	}

	return NOTIFY_DONE;
}

static struct notifier_block stats_notifier_block = {
	.notifier_call = dmw_sched_stats_event,
};

#ifdef CONFIG_DEBUG_FS
static ssize_t
dmw_sched_stats_debugfs_read(struct file *file, char __user *userbuf, size_t xcount,
                             loff_t *ppos)
{
	return simple_read_from_buffer(userbuf, xcount, ppos, stats, count * sizeof(struct sched_stat));
}

static ssize_t
dmw_sched_stats_debugfs_write(struct file *file, const char __user *userbuf,
                              size_t xcount, loff_t *ppos)
{
	struct task_struct *task;
	unsigned long flags;

	spin_lock_irqsave(&dmw_sched_stats_lock, flags);
	task = current_thread_info()->task;
	memset(stats, 0, count * sizeof(struct sched_stat));
	getnstimeofday(&stats[0].start);
	memcpy(stats[0].name, task->comm, TASK_COMM_LEN);
	stats[0].pid = task->pid;
	cur_stat = 0;
	next_stat = 1;
	spin_unlock_irqrestore(&dmw_sched_stats_lock, flags);

	return xcount;
}

static const struct file_operations dmw_sched_stats_debugfs_ops = {
	.read	= dmw_sched_stats_debugfs_read,
	.write	= dmw_sched_stats_debugfs_write,
};
#endif

static int dmw_sched_stats_init(void)
{
	struct task_struct *task = current_thread_info()->task;
	struct dentry *dentry;

	stats = vzalloc(count * sizeof(struct sched_stat));
	if (!stats)
		return -1;

	getnstimeofday(&stats[0].start);
	memcpy(stats[0].name, task->comm, TASK_COMM_LEN);
	stats[0].pid = task->pid;

	dentry = debugfs_create_file("sched_stats", S_IFREG | S_IRUGO, NULL,
	                             NULL, &dmw_sched_stats_debugfs_ops);

	thread_register_notifier(&stats_notifier_block);

	return 0;
}
late_initcall(dmw_sched_stats_init);
