#include <linux/kernel.h>
#include <linux/delay.h>
#include <asm/gpio.h>
#include <linux/platform_device.h>
#include <linux/spi/spi.h>
#include <linux/dp52.h>
#include <linux/semaphore.h>
#include <video/dw74fb.h>
#include <linux/earlysuspend.h>
#include "ili9481.h"

#ifdef TDT320_DEBUG_PROC_FS
#include <linux/proc_fs.h>
#include <asm/uaccess.h>

#define TDT320_PROC_DIR_NAME  "tdt320"
#define TDT320_PROC_FILENAME  "cmd"

static  struct proc_dir_entry *s_Tdt320ProcDirEntry;
#endif

#define LCD_GPIO_RESET GPIO_PORTB(4)

static struct spi_device *g_dw74fb_tdt320_spi_device = NULL;

struct tdt320_plat_data {

#ifdef CONFIG_HAS_EARLYSUSPEND
	struct early_suspend tdt320_early_suspend;
#endif

	char* name;
};

#ifdef TDT320_DEBUG_PROC_FS
static void ili9481_spi_write_command( unsigned char command, const unsigned char *parameters, int len );
static void tdt320t2g706_reset(void);

static int tdt320_proc_write(	struct file *File,
				const char *Buffer,
				unsigned long Count,
				void *Data)
{
	unsigned char cmds[16];	
	char buf[Count + 1];

	if (copy_from_user(buf, Buffer, 1)) {
		printk("copy_from_user failed\n");
		return -EFAULT;
	}
	buf[1] = 0;

	printk("tdt320_proc_write: value = %s\n",buf);

	if (buf[0] == '0') {
		cmds[0] = 0x66;
		ili9481_spi_write_command( ILI9481_CMD_SET_PIXEL_FORMAT, cmds, 1 );
		printk("ILI9481_CMD_SET_PIXEL_FORMAT\n");

	} else if (buf[0] == '1') {
		cmds[0] = 0x11;
		ili9481_spi_write_command( ILI9481_CMD_DISPLAY_MODE_SETTING, cmds, 1 );
		printk("ILI9481_CMD_DISPLAY_MODE_SETTING\n");

	} else if (buf[0] == '2') {
		cmds[0] = 0x81;
		ili9481_spi_write_command( ILI9481_CMD_INTERFACE_CONTROL, cmds, 1 );
		printk("ILI9481_CMD_INTERFACE_CONTROL\n");

	} else if (buf[0] == '3') {




		printk("only ILI9481_CMD_WRITE_MEMORY_START\n");
	} else if (buf[0] == '4') {
		tdt320t2g706_reset();

		ili9481_spi_write_command( ILI9481_CMD_EXIT_SLEEP_MODE, NULL, 0 );

		msleep(50);

		cmds[0] = 0x07;
		cmds[1] = 0x42;
		cmds[2] = 0x18;
		ili9481_spi_write_command( ILI9481_CMD_POWER_SETTING, cmds, 3 );

		cmds[0] = 0x00;
		cmds[1] = 0x09;
		cmds[2] = 0x0E;
		ili9481_spi_write_command( ILI9481_CMD_VCOM_CONTROL, cmds, 3 );

		cmds[0] = 0x01;
		cmds[1] = 0x02;
		ili9481_spi_write_command( ILI9481_CMD_POWER_SETTING_MODE_N, cmds, 2 );

		cmds[0] = 0x10;
		cmds[1] = 0x3B;
		cmds[2] = 0x00;
		cmds[3] = 0x02;
		cmds[4] = 0x11;
		ili9481_spi_write_command( ILI9481_CMD_PANEL_DRIVING_SETTING, cmds, 5 );

		cmds[0] = 0x10;
		cmds[1] = 0x10;
		cmds[2] = 0x22;
		ili9481_spi_write_command( ILI9481_CMD_DISPLAY_TIMING_SETTING_N, cmds, 3 );

		cmds[0] = 0x02;
		ili9481_spi_write_command( ILI9481_CMD_FRAME_RATE_INV_CONTROL, cmds, 1 );

		cmds[0] = 0x81;
		ili9481_spi_write_command( ILI9481_CMD_INTERFACE_CONTROL, cmds, 1 );

		cmds[0] = 0x00;
		cmds[1] = 0x32;
		cmds[2] = 0x36;
		cmds[3] = 0x45;
		cmds[4] = 0x06;
		cmds[5] = 0x16;
		cmds[6] = 0x37;
		cmds[7] = 0x75;
		cmds[8] = 0x77;
		cmds[9] = 0x54;
		cmds[10] = 0x0C;
		cmds[11] = 0x00;
		ili9481_spi_write_command( ILI9481_CMD_GAMMA_SETTING, cmds, 12 );

		cmds[0] = 0x00;
		cmds[1] = 0x00;
		cmds[2] = 0x01;
		cmds[3] = 0x3F;
		ili9481_spi_write_command( ILI9481_CMD_SET_COLUMN_ADDRESS, cmds, 4 );

		cmds[0] = 0x00;
		cmds[1] = 0x00;
		cmds[2] = 0x01;
		cmds[3] = 0xDF;
		ili9481_spi_write_command( ILI9481_CMD_SET_PAGE_ADDRESS, cmds, 4 );

		cmds[0] = 0x66;
		ili9481_spi_write_command( ILI9481_CMD_SET_PIXEL_FORMAT, cmds, 1 );

		cmds[0] = 0x11;
		ili9481_spi_write_command( ILI9481_CMD_DISPLAY_MODE_SETTING, cmds, 1 );

		cmds[0] = 0x9B;
		ili9481_spi_write_command( ILI9481_CMD_SET_ADRESS_MODE, cmds, 1 );

		msleep(50);

		ili9481_spi_write_command( ILI9481_CMD_ENTER_NORMAL_MODE, NULL, 0 );
		ili9481_spi_write_command( ILI9481_CMD_SET_DISPLAY_ON, NULL, 0 );

		printk("All init sequence\n");
	} else {
		printk("tdt320_proc_write: unsupported value\n");
		return -EFAULT;
	}

	ili9481_spi_write_command( ILI9481_CMD_WRITE_MEMORY_START, NULL, 0 );
	return Count;
}

static int tdt320_proc_read(	char *Page, 
				char **Start,
				off_t Off, 
				int Count, 
				int *Eof, 
				void *Data)
{
	int len;

	len = sprintf(Page,"bla bla bla\n");

	return len;
}


static int tdt320_init_proc_entry(void)
{
	struct proc_dir_entry *pEntry;

	s_Tdt320ProcDirEntry = proc_mkdir(TDT320_PROC_DIR_NAME, NULL);

	if(s_Tdt320ProcDirEntry == 0) {
		printk("Unable to create proc node: %s\n",TDT320_PROC_DIR_NAME);
		return -1;
	}

	pEntry = create_proc_entry(	TDT320_PROC_FILENAME,
					0666,
					s_Tdt320ProcDirEntry );
	if (pEntry == 0) {
		remove_proc_entry(TDT320_PROC_DIR_NAME,NULL);
		printk("Unable to create proc node: %s\n",TDT320_PROC_FILENAME);
		return -1;
	}

	pEntry->data = NULL;
	pEntry->write_proc = tdt320_proc_write;
	pEntry->read_proc  = tdt320_proc_read;

	return 0;
}
#endif //TDT320_DEBUG_PROC_FS

/*************************************************************************************************/

/* write 9bits value into array of usigned char variables, in position "index" */
static void write_9bits( unsigned int value, unsigned char *buf, unsigned int index )
{
	unsigned int	buf_index;
	unsigned char	mask;
	unsigned char	shift;
	unsigned int	first_bit;

	first_bit = index * 9;
	buf_index = first_bit / 8;
	shift = first_bit % 8;
	mask = 0xff >> shift;

//	bits_in_first_byte = 8 - shift;
//	bits_in_second_byte = 9 - bits_in_first_byte; = 9 - (8 - shift) = 9 - 8 + shift = shift + 1;

	/* clear mask in first byte */
	buf[buf_index] &= ~mask;

	/* write upper bits of value into first byte */
	buf[buf_index] |= (value >> (shift + 1)) & mask;

	/* update variable for second byte */
	buf_index++;
	shift = 7 - shift;
	mask = 0xff << shift;	// 8 - bits_in_second_byte = 8 - (shift + 1) = 7 - shift 

	/* clear mask in second byte */
	buf[buf_index] &= ~mask;

	/* write lower bits of value into second byte */
	buf[buf_index] |= (value << shift) & mask;
}

/* write command to ili9481 using 9bits SPI interface */
static void ili9481_spi_write_command( unsigned char command, const unsigned char *parameters, int len )
{
	unsigned char buf[18];
	int i;
	int index;
	const unsigned char *pparam;
	const u8 *pbuf;

	if ( !g_dw74fb_tdt320_spi_device ) {
		printk("ili9481_spi_write_command: error - global spi device variable is not set\n");
		return;
	}

	/* buffer of 18 bytes can hold 16*9bits, so total of one command and 15 parameters */
	if ( len > 15 ) {
		printk("ili9481_spi_write_command: error - too many parameters\n");
		return;
	}

	/* since the HW supports only word-len of 8 bits, we are placing our 9bits words in array */
	/* of unsigned char variables, and padding to multiples of 9x8bits with NOP command (0x0) */

	/* fill the whole buffer with 0x0 = NOP command */
	memset( buf, 0, sizeof(buf) );

	/* write the parameters into the end of the buffer */
	index = 15;
	pparam = parameters + len - 1;
	for (i = 0; i < len ; i++, index--, pparam-- ) {
		write_9bits( ILI9481_SPI_PARAMETER_BIT | *pparam, buf, index );
	}

	/* write the command before the first parameter */
	write_9bits( command, buf, index );

	/* send via spi - we are sending either 9 or 18 bytes (depends on number of parameters) */
	if ( len <= 7 ) {
		len = 9;
		pbuf = buf + 9;
	}
	else {
		len = 18;
		pbuf = buf;
	}

	spi_write( g_dw74fb_tdt320_spi_device, pbuf, len );
}

/*************************************************************************************************/

static void tdt320t2g706_reset(void)
{
	// set the gpio for reseting ili9325=>write 1
	gpio_direction_output(LCD_GPIO_RESET, 1);
	msleep(1);

	// set the gpio to get ili9325 from reset==> write 0
	gpio_set_value(LCD_GPIO_RESET,0);
	msleep(10);

	// set the gpio ili9325 write 1
	gpio_set_value(LCD_GPIO_RESET,1);
	msleep(50);
	return;
}

#ifdef CONFIG_HAS_EARLYSUSPEND

static void dw_tdt320_early_suspend(struct early_suspend *es) {

	ili9481_spi_write_command( ILI9481_CMD_SET_DISPLAY_OFF, NULL, 0 );
}

static void dw_tdt320_late_resume(struct early_suspend *es) {

	ili9481_spi_write_command( ILI9481_CMD_SET_DISPLAY_ON, NULL, 0 );
	ili9481_spi_write_command( ILI9481_CMD_WRITE_MEMORY_START, NULL, 0 );
}
#endif 


static int dw74_tdt320_init( void )
{
	unsigned char cmds[16];	

#ifdef TDT320_DEBUG_PROC_FS
	tdt320_init_proc_entry();
#endif

	/* Reset LCD */
	tdt320t2g706_reset();

	ili9481_spi_write_command( ILI9481_CMD_EXIT_SLEEP_MODE, NULL, 0 );

	msleep(50);

	cmds[0] = 0x07;
	cmds[1] = 0x42;
	cmds[2] = 0x18;
	ili9481_spi_write_command( ILI9481_CMD_POWER_SETTING, cmds, 3 );

	cmds[0] = 0x00;
	cmds[1] = 0x09;
	cmds[2] = 0x0E;
	ili9481_spi_write_command( ILI9481_CMD_VCOM_CONTROL, cmds, 3 );

	cmds[0] = 0x01;
	cmds[1] = 0x02;
	ili9481_spi_write_command( ILI9481_CMD_POWER_SETTING_MODE_N, cmds, 2 );

	cmds[0] = 0x10;
	cmds[1] = 0x3B;
	cmds[2] = 0x00;
	cmds[3] = 0x02;
	cmds[4] = 0x11;
	ili9481_spi_write_command( ILI9481_CMD_PANEL_DRIVING_SETTING, cmds, 5 );

	cmds[0] = 0x10;
	cmds[1] = 0x10;
	cmds[2] = 0x22;
	ili9481_spi_write_command( ILI9481_CMD_DISPLAY_TIMING_SETTING_N, cmds, 3 );

	cmds[0] = 0x02;
	ili9481_spi_write_command( ILI9481_CMD_FRAME_RATE_INV_CONTROL, cmds, 1 );

	cmds[0] = 0x81;
	ili9481_spi_write_command( ILI9481_CMD_INTERFACE_CONTROL, cmds, 1 );

	cmds[0] = 0x00;
	cmds[1] = 0x32;
	cmds[2] = 0x36;
	cmds[3] = 0x45;
	cmds[4] = 0x06;
	cmds[5] = 0x16;
	cmds[6] = 0x37;
	cmds[7] = 0x75;
	cmds[8] = 0x77;
	cmds[9] = 0x54;
	cmds[10] = 0x0C;
	cmds[11] = 0x00;
	ili9481_spi_write_command( ILI9481_CMD_GAMMA_SETTING, cmds, 12 );

	cmds[0] = 0x00;
	cmds[1] = 0x00;
	cmds[2] = 0x01;
	cmds[3] = 0x3F;
	ili9481_spi_write_command( ILI9481_CMD_SET_COLUMN_ADDRESS, cmds, 4 );

	cmds[0] = 0x00;
	cmds[1] = 0x00;
	cmds[2] = 0x01;
	cmds[3] = 0xDF;
	ili9481_spi_write_command( ILI9481_CMD_SET_PAGE_ADDRESS, cmds, 4 );

	cmds[0] = 0x66;
	ili9481_spi_write_command( ILI9481_CMD_SET_PIXEL_FORMAT, cmds, 1 );

	cmds[0] = 0x11;
	ili9481_spi_write_command( ILI9481_CMD_DISPLAY_MODE_SETTING, cmds, 1 );

	cmds[0] = 0x9B;
	ili9481_spi_write_command( ILI9481_CMD_SET_ADRESS_MODE, cmds, 1 );

	msleep(50);

	ili9481_spi_write_command( ILI9481_CMD_ENTER_NORMAL_MODE, NULL, 0 );
	ili9481_spi_write_command( ILI9481_CMD_SET_DISPLAY_ON, NULL, 0 );
	ili9481_spi_write_command( ILI9481_CMD_WRITE_MEMORY_START, NULL, 0 );
	
	return 0;
}

/*************************************************************************************************/


int dw74_tdt320_spi_layer_probe(struct spi_device *spi)
{
	int status;

    struct tdt320_plat_data *pdata;

	pdata = kzalloc(sizeof(struct tdt320_plat_data), GFP_KERNEL);
	if (!pdata)
		return -ENOMEM;

	spi->dev.platform_data = pdata;

	printk("%s(%d)\n",__func__,__LINE__);

	/* setup spi parameters. this makes sure that parameters we request are acceptable by spi driver */
	spi->mode = SPI_MODE_1; /* clk active high */
	spi->bits_per_word = 8;

	status = spi_setup(spi);
	if (status < 0) 
	{
		return status;
	}

	#ifdef CONFIG_HAS_EARLYSUSPEND
		pdata->tdt320_early_suspend.suspend = dw_tdt320_early_suspend;
		pdata->tdt320_early_suspend.resume = dw_tdt320_late_resume;
		pdata->tdt320_early_suspend.level = EARLY_SUSPEND_LEVEL_BLANK_SCREEN;
		register_early_suspend(&pdata->tdt320_early_suspend);
	#endif

	/* remember spi devide for later usage */
	g_dw74fb_tdt320_spi_device = spi;

	/* initialize the controller */
	dw74_tdt320_init();

	return 0;
}


static int __devexit dw74_tdt320_spi_layer_remove(struct spi_device *spi)
{
	struct tdt320_plat_data *pdata;

	pdata = spi->dev.platform_data ;

	unregister_early_suspend(&pdata->tdt320_early_suspend);

	kfree(pdata);

	return 0;
}

/*************************************************************************************************/

static struct spi_driver dw74fb_tdt320_spi_driver = {
	.driver = {
		.name =		"dw74fb_tdt320_spi",
		.bus =		&spi_bus_type,
		.owner =	THIS_MODULE,
	},
	.probe 	=	dw74_tdt320_spi_layer_probe,
	.remove =	__exit_p(dw74_tdt320_spi_layer_remove),
};

/*************************************************************************************************/

static int __init dw74_tdt320_spi_layer_init(void)
{
	return spi_register_driver(&dw74fb_tdt320_spi_driver);
}

module_init( dw74_tdt320_spi_layer_init );

/*************************************************************************************************/

